<?php
// admin/auth.php - Admin Authentication

class AdminAuth {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function login($username, $password) {
        $stmt = $this->conn->prepare("SELECT * FROM admin_users WHERE username = :username AND status = 'active'");
        $stmt->execute([':username' => $username]);
        $admin = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($admin && password_verify($password, $admin['password'])) {
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_role'] = $admin['role'];
            $_SESSION['admin_name'] = $admin['name'];
            
            // Update last login
            $this->updateLastLogin($admin['id']);
            
            // Log login activity
            $this->logActivity($admin['id'], 'login', 'Admin logged in');
            
            return true;
        }
        
        return false;
    }
    
    public function logout() {
        if (isset($_SESSION['admin_id'])) {
            $this->logActivity($_SESSION['admin_id'], 'logout', 'Admin logged out');
        }
        
        session_destroy();
        return true;
    }
    
    public function isLoggedIn() {
        return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
    }
    
    public function getAdminInfo() {
        if ($this->isLoggedIn()) {
            return [
                'id' => $_SESSION['admin_id'],
                'username' => $_SESSION['admin_username'],
                'role' => $_SESSION['admin_role'],
                'name' => $_SESSION['admin_name']
            ];
        }
        return null;
    }
    
    public function hasPermission($required_role) {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        $user_role = $_SESSION['admin_role'];
        
        // Role hierarchy: superadmin > admin > manager > staff
        $role_hierarchy = [
            'staff' => 1,
            'manager' => 2,
            'admin' => 3,
            'superadmin' => 4
        ];
        
        return isset($role_hierarchy[$user_role]) && 
               $role_hierarchy[$user_role] >= $role_hierarchy[$required_role];
    }
    
    private function updateLastLogin($admin_id) {
        $stmt = $this->conn->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = :id");
        $stmt->execute([':id' => $admin_id]);
    }
    
    private function logActivity($admin_id, $action, $description) {
        $stmt = $this->conn->prepare("
            INSERT INTO admin_activity_log 
            (admin_id, action, description, ip_address, user_agent, created_at) 
            VALUES (:admin_id, :action, :description, :ip, :agent, NOW())
        ");
        
        $stmt->execute([
            ':admin_id' => $admin_id,
            ':action' => $action,
            ':description' => $description,
            ':ip' => $_SERVER['REMOTE_ADDR'] ?? '',
            ':agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
    }
    
    public function changePassword($admin_id, $current_password, $new_password) {
        $stmt = $this->conn->prepare("SELECT password FROM admin_users WHERE id = :id");
        $stmt->execute([':id' => $admin_id]);
        $admin = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($admin && password_verify($current_password, $admin['password'])) {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            
            $stmt = $this->conn->prepare("UPDATE admin_users SET password = :password WHERE id = :id");
            $stmt->execute([
                ':password' => $hashed_password,
                ':id' => $admin_id
            ]);
            
            $this->logActivity($admin_id, 'password_change', 'Password changed');
            return true;
        }
        
        return false;
    }
}
?>