<?php
// admin/customers.php - Customers Management

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

// Create tables if they don't exist
try {
    // Create customers table if it doesn't exist
    $conn->exec("CREATE TABLE IF NOT EXISTS customers (
        id INT PRIMARY KEY AUTO_INCREMENT,
        name VARCHAR(100) NOT NULL,
        email VARCHAR(100),
        phone VARCHAR(20) NOT NULL,
        address TEXT,
        status ENUM('active', 'inactive', 'vip', 'blocked') DEFAULT 'active',
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_status (status),
        INDEX idx_phone (phone)
    )");
    
    // Create orders table if it doesn't exist
    $conn->exec("CREATE TABLE IF NOT EXISTS orders (
        id INT PRIMARY KEY AUTO_INCREMENT,
        order_number VARCHAR(50) UNIQUE NOT NULL,
        customer_id INT,
        customer_name VARCHAR(100),
        customer_phone VARCHAR(20),
        customer_address TEXT,
        total_amount DECIMAL(10,2) NOT NULL DEFAULT 0,
        status ENUM('pending', 'confirmed', 'preparing', 'delivering', 'completed', 'cancelled') DEFAULT 'pending',
        payment_method ENUM('cash', 'mpesa', 'card') DEFAULT 'cash',
        payment_status ENUM('pending', 'paid', 'failed') DEFAULT 'pending',
        delivery_location VARCHAR(100),
        delivery_fee DECIMAL(10,2) DEFAULT 0,
        notes TEXT,
        outlet VARCHAR(50) DEFAULT 'milano',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL,
        INDEX idx_customer_id (customer_id),
        INDEX idx_status (status),
        INDEX idx_outlet (outlet),
        INDEX idx_created_at (created_at)
    )");
    
} catch (Exception $e) {
    error_log("Error creating tables: " . $e->getMessage());
}

// Handle customer notes update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_notes'])) {
    try {
        $customer_id = $_POST['customer_id'];
        $notes = $_POST['notes'] ?? '';
        
        $stmt = $conn->prepare("UPDATE customers SET notes = :notes, updated_at = NOW() WHERE id = :id");
        $stmt->execute([
            ':notes' => $notes,
            ':id' => $customer_id
        ]);
        
        $_SESSION['success_message'] = "Customer notes updated successfully!";
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error updating notes: " . $e->getMessage();
    }
    header("Location: customers.php");
    exit;
}

// Handle customer status update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    try {
        $customer_id = $_POST['customer_id'];
        $status = $_POST['status'];
        
        $stmt = $conn->prepare("UPDATE customers SET status = :status, updated_at = NOW() WHERE id = :id");
        $stmt->execute([
            ':status' => $status,
            ':id' => $customer_id
        ]);
        
        // Update VIP status based on order count
        if ($status !== 'vip') {
            // Check if customer has 10+ orders
            $orderStmt = $conn->prepare("SELECT COUNT(*) as order_count FROM orders WHERE customer_id = :id AND status IN ('completed', 'delivered')");
            $orderStmt->execute([':id' => $customer_id]);
            $orderResult = $orderStmt->fetch(PDO::FETCH_ASSOC);
            
            if ($orderResult['order_count'] >= 10) {
                $stmt = $conn->prepare("UPDATE customers SET status = 'vip', updated_at = NOW() WHERE id = :id");
                $stmt->execute([':id' => $customer_id]);
            }
        }
        
        // Log activity
        $auth->logActivity($_SESSION['admin_id'], 'customer_update', "Updated customer #$customer_id status to $status");
        
        $_SESSION['success_message'] = "Customer status updated successfully!";
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error updating status: " . $e->getMessage();
    }
    header("Location: customers.php");
    exit;
}

// Handle customer deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_customer'])) {
    try {
        $customer_id = $_POST['customer_id'];
        
        // Check if customer has orders
        $stmt = $conn->prepare("SELECT COUNT(*) as order_count FROM orders WHERE customer_id = :id");
        $stmt->execute([':id' => $customer_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['order_count'] > 0) {
            $_SESSION['error_message'] = "Cannot delete customer with existing orders. Mark as inactive instead.";
        } else {
            $stmt = $conn->prepare("DELETE FROM customers WHERE id = :id");
            $stmt->execute([':id' => $customer_id]);
            
            // Log activity
            $auth->logActivity($_SESSION['admin_id'], 'customer_delete', "Deleted customer #$customer_id");
            
            $_SESSION['success_message'] = "Customer deleted successfully!";
        }
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error deleting customer: " . $e->getMessage();
    }
    header("Location: customers.php");
    exit;
}

// Handle search and filters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'revenue_desc';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$min_orders = $_GET['min_orders'] ?? '';
$outlet = $_GET['outlet'] ?? 'milano'; // Default to milano

// Build base query
$query = "SELECT 
            c.*,
            COUNT(o.id) as total_orders,
            SUM(CASE WHEN o.status IN ('completed', 'delivered') THEN o.total_amount ELSE 0 END) as total_revenue,
            MAX(o.created_at) as last_order_date,
            MIN(o.created_at) as first_order_date
          FROM customers c
          LEFT JOIN orders o ON c.id = o.customer_id AND o.outlet = :outlet
          WHERE 1=1";
          
$params = [':outlet' => $outlet];

if (!empty($search)) {
    $query .= " AND (c.name LIKE :search OR c.email LIKE :search OR c.phone LIKE :search)";
    $params[':search'] = "%$search%";
}

if (!empty($status_filter) && in_array($status_filter, ['active', 'inactive', 'vip', 'blocked'])) {
    $query .= " AND c.status = :status";
    $params[':status'] = $status_filter;
}

// Group by customer
$query .= " GROUP BY c.id";

// Having clause for minimum orders
if (!empty($min_orders) && is_numeric($min_orders)) {
    $query .= " HAVING total_orders >= :min_orders";
    $params[':min_orders'] = (int)$min_orders;
}

// Apply date filters in HAVING clause (for last order date)
if (!empty($date_from)) {
    $query .= " AND (MAX(o.created_at) IS NULL OR DATE(MAX(o.created_at)) >= :date_from)";
    $params[':date_from'] = $date_from;
}

if (!empty($date_to)) {
    $query .= " AND (MAX(o.created_at) IS NULL OR DATE(MAX(o.created_at)) <= :date_to)";
    $params[':date_to'] = $date_to;
}

// Sorting
$sort_options = [
    'name_asc' => 'c.name ASC',
    'name_desc' => 'c.name DESC',
    'revenue_desc' => 'total_revenue DESC',
    'revenue_asc' => 'total_revenue ASC',
    'orders_desc' => 'total_orders DESC',
    'orders_asc' => 'total_orders ASC',
    'recent' => 'last_order_date DESC',
    'oldest' => 'first_order_date ASC'
];

$sort_clause = $sort_options[$sort_by] ?? 'total_revenue DESC';
$query .= " ORDER BY $sort_clause";

// Pagination
$page = $_GET['page'] ?? 1;
$limit = 25;
$offset = ($page - 1) * $limit;

// Count total (using a separate query for complex aggregation)
$count_query = "SELECT COUNT(DISTINCT c.id) as total FROM customers c 
                LEFT JOIN orders o ON c.id = o.customer_id AND o.outlet = :outlet WHERE 1=1";
$count_params = [':outlet' => $outlet];

if (!empty($search)) {
    $count_query .= " AND (c.name LIKE :search OR c.email LIKE :search OR c.phone LIKE :search)";
    $count_params[':search'] = "%$search%";
}

if (!empty($status_filter) && in_array($status_filter, ['active', 'inactive', 'vip', 'blocked'])) {
    $count_query .= " AND c.status = :status";
    $count_params[':status'] = $status_filter;
}

try {
    $stmt = $conn->prepare($count_query);
    $stmt->execute($count_params);
    $total_rows = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    $total_pages = ceil($total_rows / $limit);
} catch (Exception $e) {
    $total_rows = 0;
    $total_pages = 1;
    error_log("Error counting customers: " . $e->getMessage());
}

// Get customers with limit
$query .= " LIMIT $limit OFFSET $offset";

try {
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $customers = [];
    error_log("Error fetching customers: " . $e->getMessage());
}

// Get statistics for dashboard
$stats = [];
$status_counts = [];

try {
    // Total customers
    $stmt = $conn->query("SELECT COUNT(*) as total FROM customers");
    $stats['total_customers'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

    // Active customers (with orders in last 30 days)
    $stmt = $conn->query("SELECT COUNT(DISTINCT customer_id) as total FROM orders 
                          WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND outlet = '$outlet'");
    $stats['active_customers'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

    // Total revenue
    $stmt = $conn->query("SELECT SUM(total_amount) as total FROM orders 
                          WHERE status IN ('completed', 'delivered') AND outlet = '$outlet'");
    $stats['total_revenue'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

    // Average order value
    $stmt = $conn->query("SELECT AVG(total_amount) as avg FROM orders 
                          WHERE status IN ('completed', 'delivered') AND outlet = '$outlet'");
    $stats['avg_order_value'] = $stmt->fetch(PDO::FETCH_ASSOC)['avg'] ?? 0;

    // Get status counts
    $stmt = $conn->query("SELECT status, COUNT(*) as count FROM customers GROUP BY status");
    $status_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($status_data as $row) {
        $status_counts[$row['status']] = $row['count'];
    }

    // Get top customers by revenue (for sidebar)
    $top_customers_query = "SELECT c.id, c.name, c.email, 
                           SUM(o.total_amount) as revenue,
                           COUNT(o.id) as order_count
                           FROM customers c
                           JOIN orders o ON c.id = o.customer_id
                           WHERE o.status IN ('completed', 'delivered') AND o.outlet = :outlet
                           GROUP BY c.id
                           ORDER BY revenue DESC
                           LIMIT 5";
    $top_customers_stmt = $conn->prepare($top_customers_query);
    $top_customers_stmt->execute([':outlet' => $outlet]);
    $top_customers = $top_customers_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching statistics: " . $e->getMessage());
}

// Define customer statuses
$customer_statuses = ['active', 'inactive', 'vip', 'blocked'];

// Check for messages from session
$success = $_SESSION['success_message'] ?? '';
$error = $_SESSION['error_message'] ?? '';
unset($_SESSION['success_message']);
unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customers Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        :root {
            --primary: #D35400;
            --primary-dark: #A04000;
            --secondary: #27AE60;
        }
        
        .stats-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
            transition: transform 0.2s;
        }
        
        .stats-card:hover {
            transform: translateY(-2px);
        }
        
        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            margin-bottom: 15px;
        }
        
        .customer-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
            transition: transform 0.2s;
            border: 1px solid #eee;
        }
        
        .customer-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .customer-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            font-weight: bold;
        }
        
        .badge-vip {
            background: linear-gradient(135deg, #FFD700, #FFA500);
            color: #333;
        }
        
        .badge-active { background: var(--secondary); color: white; }
        .badge-inactive { background: #6c757d; color: white; }
        .badge-blocked { background: #dc3545; color: white; }
        
        .revenue-badge {
            background: #2e7d32;
            color: white;
            padding: 5px 10px;
            border-radius: 20px;
            font-weight: bold;
        }
        
        .notes-textarea {
            resize: vertical;
            min-height: 100px;
        }
        
        .customer-timeline {
            border-left: 3px solid var(--primary);
            padding-left: 15px;
            margin-left: 8px;
        }
        
        .timeline-item {
            margin-bottom: 15px;
            position: relative;
        }
        
        .timeline-item:before {
            content: '';
            position: absolute;
            left: -21px;
            top: 5px;
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: var(--primary);
        }
        
        .filter-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
        }
        
        .progress-bar-custom {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
        }
        
        .outlet-selector {
            margin-bottom: 20px;
        }
        
        .outlet-selector .btn-group {
            width: 100%;
        }
        
        .outlet-selector .btn {
            flex: 1;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-9">
                <?php if ($success): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle"></i> <?php echo $success; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- Outlet Selector -->
                <div class="outlet-selector">
                    <div class="btn-group" role="group">
                        <a href="?outlet=milano" class="btn <?php echo $outlet === 'milano' ? 'btn-warning' : 'btn-outline-warning'; ?>">
                            <i class="fas fa-pizza-slice"></i> Milano Pizza
                        </a>
                        <a href="?outlet=sushi" class="btn <?php echo $outlet === 'sushi' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-fish"></i> Sushi Bar
                        </a>
                    </div>
                </div>
                
                <!-- Statistics Dashboard -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e3f2fd; color: #1976d2;">
                                <i class="fas fa-users"></i>
                            </div>
                            <h3><?php echo number_format($stats['total_customers']); ?></h3>
                            <p class="text-muted">Total Customers</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e8f5e9; color: var(--secondary);">
                                <i class="fas fa-user-check"></i>
                            </div>
                            <h3><?php echo number_format($stats['active_customers']); ?></h3>
                            <p class="text-muted">Active (30 days)</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #fff3e0; color: var(--primary);">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <h3>KES <?php echo number_format($stats['total_revenue'], 2); ?></h3>
                            <p class="text-muted">Total Revenue</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #f3e5f5; color: #7b1fa2;">
                                <i class="fas fa-shopping-cart"></i>
                            </div>
                            <h3>KES <?php echo number_format($stats['avg_order_value'], 2); ?></h3>
                            <p class="text-muted">Avg Order Value</p>
                        </div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="filter-card">
                    <form method="GET" action="" class="row g-3">
                        <input type="hidden" name="outlet" value="<?php echo $outlet; ?>">
                        
                        <div class="col-md-3">
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-search"></i></span>
                                <input type="text" 
                                       class="form-control" 
                                       name="search" 
                                       placeholder="Search customers..."
                                       value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <select class="form-select" name="status">
                                <option value="">All Status</option>
                                <?php foreach ($customer_statuses as $status): ?>
                                <option value="<?php echo $status; ?>" 
                                    <?php echo $status_filter === $status ? 'selected' : ''; ?>>
                                    <?php echo ucfirst($status); ?> 
                                    (<?php echo $status_counts[$status] ?? 0; ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="col-md-2">
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-shopping-cart"></i></span>
                                <input type="number" 
                                       class="form-control" 
                                       name="min_orders" 
                                       value="<?php echo htmlspecialchars($min_orders); ?>"
                                       placeholder="Min Orders"
                                       min="0">
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <select class="form-select" name="sort_by">
                                <option value="revenue_desc" <?php echo $sort_by === 'revenue_desc' ? 'selected' : ''; ?>>Highest Revenue</option>
                                <option value="revenue_asc" <?php echo $sort_by === 'revenue_asc' ? 'selected' : ''; ?>>Lowest Revenue</option>
                                <option value="orders_desc" <?php echo $sort_by === 'orders_desc' ? 'selected' : ''; ?>>Most Orders</option>
                                <option value="orders_asc" <?php echo $sort_by === 'orders_asc' ? 'selected' : ''; ?>>Fewest Orders</option>
                                <option value="name_asc" <?php echo $sort_by === 'name_asc' ? 'selected' : ''; ?>>Name A-Z</option>
                                <option value="name_desc" <?php echo $sort_by === 'name_desc' ? 'selected' : ''; ?>>Name Z-A</option>
                                <option value="recent" <?php echo $sort_by === 'recent' ? 'selected' : ''; ?>>Recent Activity</option>
                                <option value="oldest" <?php echo $sort_by === 'oldest' ? 'selected' : ''; ?>>Oldest Customer</option>
                            </select>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-filter"></i> Filter
                                </button>
                                <a href="customers.php?outlet=<?php echo $outlet; ?>" class="btn btn-outline-secondary">
                                    <i class="fas fa-redo"></i> Reset
                                </a>
                            </div>
                        </div>
                    </form>
                    
                    <!-- Date Range Filters -->
                    <form method="GET" action="" class="row g-3 mt-3">
                        <input type="hidden" name="outlet" value="<?php echo $outlet; ?>">
                        <input type="hidden" name="search" value="<?php echo htmlspecialchars($search); ?>">
                        <input type="hidden" name="status" value="<?php echo htmlspecialchars($status_filter); ?>">
                        <input type="hidden" name="min_orders" value="<?php echo htmlspecialchars($min_orders); ?>">
                        <input type="hidden" name="sort_by" value="<?php echo htmlspecialchars($sort_by); ?>">
                        
                        <div class="col-md-3">
                            <label class="form-label small">From Date</label>
                            <input type="date" 
                                   class="form-control form-control-sm" 
                                   name="date_from" 
                                   value="<?php echo htmlspecialchars($date_from); ?>">
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label small">To Date</label>
                            <input type="date" 
                                   class="form-control form-control-sm" 
                                   name="date_to" 
                                   value="<?php echo htmlspecialchars($date_to); ?>">
                        </div>
                        
                        <div class="col-md-2">
                            <label class="form-label small">&nbsp;</label>
                            <button type="submit" class="btn btn-sm btn-outline-primary w-100">
                                <i class="fas fa-calendar-alt"></i> Apply Dates
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Customers List -->
                <div class="row">
                    <?php if (empty($customers)): ?>
                    <div class="col-12">
                        <div class="card">
                            <div class="card-body text-center py-5">
                                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                <h4>No customers found</h4>
                                <p class="text-muted">Try adjusting your search filters or add new customers</p>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <?php foreach ($customers as $customer): 
                        $initial = !empty($customer['name']) ? strtoupper(substr($customer['name'], 0, 1)) : '?';
                        $last_order = $customer['last_order_date'] ? date('M j, Y', strtotime($customer['last_order_date'])) : 'Never';
                        $first_order = $customer['first_order_date'] ? date('M j, Y', strtotime($customer['first_order_date'])) : 'No orders';
                        $customer_name = htmlspecialchars($customer['name'] ?? 'Unknown');
                        $customer_email = htmlspecialchars($customer['email'] ?? '');
                        $customer_phone = htmlspecialchars($customer['phone'] ?? '');
                        $customer_address = htmlspecialchars($customer['address'] ?? '');
                        $customer_status = $customer['status'] ?? 'active';
                        $customer_notes = $customer['notes'] ?? '';
                        $total_orders = $customer['total_orders'] ?? 0;
                        $total_revenue = $customer['total_revenue'] ?? 0;
                        $avg_order = $total_orders > 0 ? $total_revenue / $total_orders : 0;
                    ?>
                    <div class="col-md-4">
                        <div class="customer-card">
                            <div class="d-flex justify-content-between align-items-start mb-3">
                                <div class="d-flex align-items-center">
                                    <div class="customer-avatar me-3">
                                        <?php echo $initial; ?>
                                    </div>
                                    <div>
                                        <h5 class="mb-1"><?php echo $customer_name; ?></h5>
                                        <span class="badge badge-<?php echo $customer_status; ?>">
                                            <?php echo ucfirst($customer_status); ?>
                                        </span>
                                        <?php if ($total_orders >= 10): ?>
                                        <span class="badge badge-vip ms-1">VIP</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="dropdown">
                                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" 
                                            type="button" 
                                            data-bs-toggle="dropdown">
                                        <i class="fas fa-ellipsis-v"></i>
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="#" 
                                               data-bs-toggle="modal" 
                                               data-bs-target="#notesModal<?php echo $customer['id']; ?>">
                                            <i class="fas fa-edit me-2"></i>Edit Notes
                                        </a></li>
                                        <li><a class="dropdown-item" href="orders.php?search=<?php echo urlencode($customer_phone); ?>">
                                            <i class="fas fa-list me-2"></i>View Orders
                                        </a></li>
                                        <li><a class="dropdown-item" href="#" 
                                               data-bs-toggle="modal" 
                                               data-bs-target="#statusModal<?php echo $customer['id']; ?>">
                                            <i class="fas fa-sync me-2"></i>Change Status
                                        </a></li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li>
                                            <form method="POST" action="" class="d-inline">
                                                <input type="hidden" name="customer_id" value="<?php echo $customer['id']; ?>">
                                                <button type="submit" 
                                                        name="delete_customer" 
                                                        class="dropdown-item text-danger"
                                                        onclick="return confirm('Are you sure you want to delete this customer?')">
                                                    <i class="fas fa-trash me-2"></i>Delete
                                                </button>
                                            </form>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <?php if ($customer_email): ?>
                                <p class="mb-1">
                                    <i class="fas fa-envelope me-2 text-muted"></i>
                                    <a href="mailto:<?php echo $customer_email; ?>" class="text-decoration-none">
                                        <?php echo $customer_email; ?>
                                    </a>
                                </p>
                                <?php endif; ?>
                                
                                <p class="mb-2">
                                    <i class="fas fa-phone me-2 text-muted"></i>
                                    <a href="tel:<?php echo $customer_phone; ?>" class="text-decoration-none">
                                        <?php echo $customer_phone; ?>
                                    </a>
                                </p>
                                
                                <?php if ($customer_address): ?>
                                <p class="mb-2 text-muted small">
                                    <i class="fas fa-map-marker-alt me-2"></i>
                                    <?php echo substr($customer_address, 0, 50); ?>...
                                </p>
                                <?php endif; ?>
                            </div>
                            
                            <div class="row text-center mb-3">
                                <div class="col-4">
                                    <div class="border rounded p-2">
                                        <h6 class="mb-0 text-primary"><?php echo $total_orders; ?></h6>
                                        <small class="text-muted">Orders</small>
                                    </div>
                                </div>
                                <div class="col-4">
                                    <div class="border rounded p-2">
                                        <h6 class="mb-0 text-success">KES <?php echo number_format($total_revenue, 2); ?></h6>
                                        <small class="text-muted">Revenue</small>
                                    </div>
                                </div>
                                <div class="col-4">
                                    <div class="border rounded p-2">
                                        <h6 class="mb-0 text-warning">
                                            KES <?php echo number_format($avg_order, 2); ?>
                                        </h6>
                                        <small class="text-muted">Avg Order</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="customer-timeline small">
                                <div class="timeline-item">
                                    <strong>First Order:</strong>
                                    <span class="text-muted"><?php echo $first_order; ?></span>
                                </div>
                                <div class="timeline-item">
                                    <strong>Last Order:</strong>
                                    <span class="text-muted"><?php echo $last_order; ?></span>
                                </div>
                            </div>
                            
                            <?php if ($customer_notes): ?>
                            <div class="mt-3 p-2 bg-light rounded">
                                <small class="text-muted">Notes:</small>
                                <p class="mb-0 small"><?php echo nl2br($customer_notes); ?></p>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Notes Modal -->
                        <div class="modal fade" id="notesModal<?php echo $customer['id']; ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="POST" action="">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Edit Notes - <?php echo $customer_name; ?></h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="customer_id" value="<?php echo $customer['id']; ?>">
                                            <div class="mb-3">
                                                <label class="form-label">Customer Notes</label>
                                                <textarea class="form-control notes-textarea" 
                                                          name="notes" 
                                                          rows="4"><?php echo $customer_notes; ?></textarea>
                                                <div class="form-text">Add any special notes about this customer</div>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" name="update_notes" class="btn btn-primary">Save Notes</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Status Modal -->
                        <div class="modal fade" id="statusModal<?php echo $customer['id']; ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="POST" action="">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Change Status - <?php echo $customer_name; ?></h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="customer_id" value="<?php echo $customer['id']; ?>">
                                            <div class="mb-3">
                                                <label class="form-label">Select Status</label>
                                                <select class="form-select" name="status" required>
                                                    <?php foreach ($customer_statuses as $status): ?>
                                                    <option value="<?php echo $status; ?>" 
                                                        <?php echo $customer_status === $status ? 'selected' : ''; ?>>
                                                        <?php echo ucfirst($status); ?>
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            <div class="alert alert-info small">
                                                <i class="fas fa-info-circle"></i>
                                                VIP status is automatically assigned to customers with 10+ orders.
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" name="update_status" class="btn btn-primary">Update Status</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav class="mt-4">
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?<?php 
                                echo http_build_query(array_merge(
                                    $_GET,
                                    ['page' => $page - 1]
                                ));
                            ?>">
                                <i class="fas fa-chevron-left"></i> Previous
                            </a>
                        </li>
                        
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?<?php 
                                echo http_build_query(array_merge(
                                    $_GET,
                                    ['page' => $i]
                                ));
                            ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                        <?php endfor; ?>
                        
                        <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?<?php 
                                echo http_build_query(array_merge(
                                    $_GET,
                                    ['page' => $page + 1]
                                ));
                            ?>">
                                Next <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
            
            <!-- Sidebar -->
            <div class="col-md-3">
                <!-- Top Customers -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                        <span><i class="fas fa-crown me-2"></i> Top Customers</span>
                        <small><?php echo $outlet === 'milano' ? 'Pizza' : 'Sushi'; ?></small>
                    </div>
                    <div class="card-body p-0">
                        <div class="list-group list-group-flush">
                            <?php if (empty($top_customers)): ?>
                            <div class="list-group-item text-center py-3">
                                <i class="fas fa-chart-line fa-2x text-muted mb-2"></i>
                                <p class="mb-0 text-muted small">No customer data yet</p>
                            </div>
                            <?php endif; ?>
                            
                            <?php foreach ($top_customers as $index => $top_customer): ?>
                            <a href="orders.php?search=<?php echo urlencode($top_customer['email'] ?? $top_customer['name']); ?>" 
                               class="list-group-item list-group-item-action">
                                <div class="d-flex w-100 justify-content-between">
                                    <div>
                                        <span class="badge bg-primary me-2">#<?php echo $index + 1; ?></span>
                                        <strong><?php echo htmlspecialchars($top_customer['name'] ?? 'Unknown'); ?></strong>
                                    </div>
                                    <span class="badge bg-success">
                                        KES <?php echo number_format($top_customer['revenue'] ?? 0, 2); ?>
                                    </span>
                                </div>
                                <small class="text-muted">
                                    <?php echo $top_customer['order_count'] ?? 0; ?> orders
                                </small>
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Customer Analytics -->
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span><i class="fas fa-chart-pie me-2"></i> Customer Analytics</span>
                        <small>Total: <?php echo $stats['total_customers']; ?></small>
                    </div>
                    <div class="card-body">
                        <!-- Status Distribution -->
                        <?php foreach ($customer_statuses as $status): 
                            $count = $status_counts[$status] ?? 0;
                            if ($stats['total_customers'] > 0):
                                $percentage = ($count / $stats['total_customers']) * 100;
                        ?>
                        <div class="mb-3">
                            <div class="d-flex justify-content-between mb-1">
                                <span><?php echo ucfirst($status); ?></span>
                                <span><?php echo $count; ?> (<?php echo round($percentage, 1); ?>%)</span>
                            </div>
                            <div class="progress" style="height: 8px;">
                                <div class="progress-bar progress-bar-custom" 
                                     style="width: <?php echo $percentage; ?>%"></div>
                            </div>
                        </div>
                        <?php endif;
                        endforeach; ?>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="card">
                    <div class="card-header">
                        <i class="fas fa-bolt me-2"></i> Quick Actions
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <a href="new_order.php" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i> Create New Order
                            </a>
                            <button class="btn btn-outline-primary" onclick="exportCustomers()">
                                <i class="fas fa-download me-2"></i> Export Customers
                            </button>
                            <a href="reports.php?report=customers&outlet=<?php echo $outlet; ?>" class="btn btn-outline-secondary">
                                <i class="fas fa-chart-bar me-2"></i> View Reports
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Export customers function
        function exportCustomers() {
            const params = new URLSearchParams(window.location.search);
            const outlet = params.get('outlet') || 'milano';
            
            // Create export URL with current filters
            let exportUrl = 'export_customers.php?' + params.toString();
            
            // Open in new tab
            window.open(exportUrl, '_blank');
        }
        
        // Auto-refresh stats every 60 seconds
        setInterval(function() {
            if (!document.hidden) {
                // Refresh the page to update stats
                // You could implement AJAX refresh here if needed
                console.log('Stats refresh triggered');
            }
        }, 60000);
        
        // Quick search
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + F to focus search
            if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
                e.preventDefault();
                document.querySelector('input[name="search"]').focus();
            }
        });
        
        // Initialize tooltips
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    </script>
</body>
</html>