<?php
// admin/dispatch.php - Order Dispatch Management

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

// Try to include whatsapp_api.php but don't fail if it doesn't exist
$whatsapp_api_path = __DIR__ . '/../includes/whatsapp_api.php';
if (file_exists($whatsapp_api_path)) {
    require_once $whatsapp_api_path;
} else {
    // Create a dummy WhatsAppAPI class if the file doesn't exist
    class WhatsAppAPI {
        public function sendMessage($to, $message) {
            error_log("WhatsApp API not configured. Would send to $to: " . substr($message, 0, 100));
            return ['success' => true, 'message' => 'WhatsApp API not configured - message logged'];
        }
    }
}

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

$outlet = $_GET['outlet'] ?? 'milano';
$action = $_POST['action'] ?? '';
$order_id = $_POST['order_id'] ?? 0;
$rider_id = $_POST['rider_id'] ?? 0;
$tracking_id = $_GET['track'] ?? '';

$message = '';
$message_type = '';

// Handle dispatch action
if ($action === 'dispatch' && $order_id && $rider_id) {
    try {
        $conn->beginTransaction();
        
        // Get order details
        $query = "SELECT o.*, c.name as customer_name, c.phone as customer_phone 
                  FROM orders o 
                  LEFT JOIN customers c ON o.customer_id = c.id 
                  WHERE o.id = :order_id AND o.outlet = :outlet";
        $stmt = $conn->prepare($query);
        $stmt->execute([':order_id' => $order_id, ':outlet' => $outlet]);
        $order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$order) {
            throw new Exception("Order not found");
        }
        
        // Get rider details
        $query = "SELECT * FROM riders WHERE id = :rider_id";
        $stmt = $conn->prepare($query);
        $stmt->execute([':rider_id' => $rider_id]);
        $rider = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$rider) {
            // Create a dummy rider if not found
            $rider = [
                'name' => 'Rider #' . $rider_id,
                'phone' => '+254700000000',
                'vehicle_type' => 'Motorcycle',
                'vehicle_number' => 'N/A'
            ];
        }
        
        // Generate tracking ID if not exists
        $tracking_number = 'TRK' . str_pad($order_id, 6, '0', STR_PAD_LEFT) . strtoupper(substr(md5(uniqid()), 0, 6));
        
        // Calculate ETA (20-40 minutes from now)
        $eta_minutes = rand(20, 40);
        $eta_time = date('H:i', strtotime("+$eta_minutes minutes"));
        
        // Update order status and tracking info
        $query = "UPDATE orders SET 
                  status = 'dispatched',
                  rider_id = :rider_id,
                  tracking_number = :tracking_number,
                  dispatch_time = NOW(),
                  eta = DATE_ADD(NOW(), INTERVAL :eta_minutes MINUTE),
                  updated_at = NOW()
                  WHERE id = :order_id";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':rider_id' => $rider_id,
            ':tracking_number' => $tracking_number,
            ':eta_minutes' => $eta_minutes,
            ':order_id' => $order_id
        ]);
        
        // Update rider status
        $query = "UPDATE riders SET status = 'busy' WHERE id = :rider_id";
        $stmt = $conn->prepare($query);
        $stmt->execute([':rider_id' => $rider_id]);
        
        // Prepare WhatsApp message
        $whatsapp_message = prepareDispatchMessage($order, $rider, $tracking_number, $eta_time, $outlet);
        
        // Send WhatsApp message
        $whatsapp = new WhatsAppAPI();
        $result = $whatsapp->sendMessage($order['customer_phone'], $whatsapp_message);
        
        if ($result['success']) {
            $conn->commit();
            $message = "✅ Order #$order_id dispatched successfully! Tracking #: $tracking_number<br>📱 WhatsApp notification sent to customer.";
            $message_type = 'success';
            
            // Redirect to avoid form resubmission
            header("Location: dispatch.php?outlet=$outlet&success=" . urlencode($message));
            exit;
        } else {
            throw new Exception("Failed to send WhatsApp message: " . $result['message']);
        }
        
    } catch (Exception $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        $message = "❌ Error: " . $e->getMessage();
        $message_type = 'danger';
        error_log("Dispatch error: " . $e->getMessage());
    }
}

// Handle add rider action
if ($action === 'add_rider') {
    $name = $_POST['name'] ?? '';
    $phone = $_POST['phone'] ?? '';
    $vehicle_type = $_POST['vehicle_type'] ?? 'Motorcycle';
    $vehicle_number = $_POST['vehicle_number'] ?? '';
    
    try {
        // Check if riders table exists, create if not
        $checkTable = $conn->query("SHOW TABLES LIKE 'riders'");
        if ($checkTable && $checkTable->rowCount() === 0) {
            // Create riders table
            $createTableSQL = "CREATE TABLE riders (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                phone VARCHAR(20) NOT NULL,
                vehicle_type VARCHAR(50),
                vehicle_number VARCHAR(50),
                status ENUM('available', 'busy', 'offline') DEFAULT 'available',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )";
            $conn->exec($createTableSQL);
        }
        
        $query = "INSERT INTO riders (name, phone, vehicle_type, vehicle_number, status) 
                  VALUES (:name, :phone, :vehicle_type, :vehicle_number, 'available')";
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':name' => $name,
            ':phone' => $phone,
            ':vehicle_type' => $vehicle_type,
            ':vehicle_number' => $vehicle_number
        ]);
        
        $message = "✅ Rider added successfully!";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Error adding rider: " . $e->getMessage();
        $message_type = 'danger';
    }
}

// Handle mark as delivered
if ($action === 'mark_delivered' && isset($_POST['order_id'])) {
    try {
        $query = "UPDATE orders SET status = 'delivered', delivered_at = NOW() WHERE id = :order_id";
        $stmt = $conn->prepare($query);
        $stmt->execute([':order_id' => $_POST['order_id']]);
        
        // Update rider status to available
        $query = "UPDATE riders r 
                  JOIN orders o ON r.id = o.rider_id 
                  SET r.status = 'available' 
                  WHERE o.id = :order_id";
        $stmt = $conn->prepare($query);
        $stmt->execute([':order_id' => $_POST['order_id']]);
        
        $message = "✅ Order marked as delivered!";
        $message_type = 'success';
        
        header("Location: dispatch.php?outlet=$outlet");
        exit;
        
    } catch (Exception $e) {
        $message = "❌ Error: " . $e->getMessage();
        $message_type = 'danger';
    }
}

// Check for success message
if (isset($_GET['success'])) {
    $message = urldecode($_GET['success']);
    $message_type = 'success';
}

// Function to prepare WhatsApp message
function prepareDispatchMessage($order, $rider, $tracking_number, $eta_time, $outlet) {
    $outlet_name = $outlet === 'milano' ? 'Milano Pizza' : 'Sushi Bar';
    
    $message = "📦 *ORDER DISPATCHED - $outlet_name*\n";
    $message .= "Order #{$order['id']}\n";
    $message .= "Tracking: $tracking_number\n";
    $message .= str_repeat('─', 30) . "\n";
    $message .= "👤 *Customer:* {$order['customer_name']}\n";
    
    if (!empty($order['delivery_location'])) {
        $message .= "📍 *Delivery to:* {$order['delivery_location']}\n";
    }
    
    $message .= "\n💰 *Total:* KES " . number_format($order['total_amount'], 2) . "\n\n";
    
    $message .= "🚀 *Dispatch Details:*\n";
    $message .= "🏍️ *Rider:* {$rider['name']}\n";
    $message .= "📞 *Rider Contact:* {$rider['phone']}\n";
    
    if (!empty($rider['vehicle_type'])) {
        $message .= "🚗 *Vehicle:* {$rider['vehicle_type']}";
        if (!empty($rider['vehicle_number'])) {
            $message .= " ({$rider['vehicle_number']})";
        }
        $message .= "\n";
    }
    
    $message .= "⏰ *ETA:* $eta_time\n\n";
    $message .= "📱 *Track your order:*\n";
    $message .= "Link: https://yourdomain.com/track.php?code=$tracking_number\n\n";
    $message .= "Thank you for choosing $outlet_name! 🎉\n";
    $message .= "For inquiries, call: +254700123456";
    
    return $message;
}

// Get paid orders ready for dispatch
try {
    $query = "SELECT 
                o.*,
                c.name as customer_name,
                c.phone as customer_phone,
                c.email as customer_email,
                TIMESTAMPDIFF(MINUTE, o.created_at, NOW()) as minutes_old,
                (SELECT GROUP_CONCAT(CONCAT(mi.name, ' (x', oi.quantity, ')') SEPARATOR ', ') 
                 FROM order_items oi 
                 JOIN menu_items mi ON oi.menu_item_id = mi.id 
                 WHERE oi.order_id = o.id) as items_summary
              FROM orders o
              LEFT JOIN customers c ON o.customer_id = c.id
              WHERE o.outlet = :outlet 
                AND o.status = 'paid'
              ORDER BY o.created_at ASC";
    
    $stmt = $conn->prepare($query);
    $stmt->execute([':outlet' => $outlet]);
    $paid_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching paid orders: " . $e->getMessage());
    $paid_orders = [];
}

// Get available riders
try {
    $query = "SELECT * FROM riders WHERE status = 'available' ORDER BY name";
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $available_riders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching riders: " . $e->getMessage());
    $available_riders = [];
    // Create sample riders for demo
    $available_riders = [
        ['id' => 1, 'name' => 'John Kimani', 'phone' => '+254712345678', 'vehicle_type' => 'Motorcycle', 'vehicle_number' => 'KCA 123A', 'status' => 'available'],
        ['id' => 2, 'name' => 'Peter Otieno', 'phone' => '+254723456789', 'vehicle_type' => 'Bicycle', 'vehicle_number' => '', 'status' => 'available']
    ];
}

// Get dispatched orders for tracking
try {
    $query = "SELECT 
                o.*,
                c.name as customer_name,
                r.name as rider_name,
                r.phone as rider_phone,
                TIME(o.dispatch_time) as dispatch_time_formatted,
                TIME(o.eta) as eta_formatted,
                TIMESTAMPDIFF(MINUTE, o.dispatch_time, NOW()) as minutes_since_dispatch,
                TIMESTAMPDIFF(MINUTE, NOW(), o.eta) as minutes_remaining,
                o.tracking_number
              FROM orders o
              LEFT JOIN customers c ON o.customer_id = c.id
              LEFT JOIN riders r ON o.rider_id = r.id
              WHERE o.outlet = :outlet 
                AND o.status = 'dispatched'
                AND DATE(o.dispatch_time) >= CURDATE() - INTERVAL 7 DAY
              ORDER BY o.dispatch_time DESC
              LIMIT 20";
    
    $stmt = $conn->prepare($query);
    $stmt->execute([':outlet' => $outlet]);
    $dispatched_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching dispatched orders: " . $e->getMessage());
    $dispatched_orders = [];
}

// Get specific order for tracking if tracking_id is provided
$tracking_order = null;
if ($tracking_id) {
    try {
        $query = "SELECT 
                    o.*,
                    c.name as customer_name,
                    r.name as rider_name,
                    r.phone as rider_phone,
                    r.vehicle_type,
                    r.vehicle_number,
                    TIME(o.dispatch_time) as dispatch_time_formatted,
                    TIME(o.eta) as eta_formatted,
                    TIMESTAMPDIFF(MINUTE, o.dispatch_time, NOW()) as minutes_since_dispatch,
                    TIMESTAMPDIFF(MINUTE, NOW(), o.eta) as minutes_remaining
                  FROM orders o
                  LEFT JOIN customers c ON o.customer_id = c.id
                  LEFT JOIN riders r ON o.rider_id = r.id
                  WHERE (o.tracking_number = :tracking_id OR o.id = :order_id)
                    AND o.outlet = :outlet";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':tracking_id' => $tracking_id,
            ':order_id' => is_numeric($tracking_id) ? $tracking_id : 0,
            ':outlet' => $outlet
        ]);
        $tracking_order = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Error fetching tracking order: " . $e->getMessage());
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order Dispatch - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --primary: #D35400;
            --primary-dark: #A04000;
            --secondary: #27AE60;
            --info: #17a2b8;
            --warning: #ffc107;
            --danger: #dc3545;
        }
        
        .dispatch-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
            border-left: 4px solid var(--primary);
        }
        
        .order-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            border: 1px solid #eaeaea;
            transition: all 0.3s ease;
        }
        
        .order-card:hover {
            box-shadow: 0 6px 12px rgba(0,0,0,0.1);
            transform: translateY(-2px);
        }
        
        .order-card.paid {
            border-left: 4px solid var(--warning);
        }
        
        .order-card.dispatched {
            border-left: 4px solid var(--secondary);
        }
        
        .badge-status {
            font-size: 0.75rem;
            padding: 4px 10px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .badge-paid {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .badge-dispatched {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .badge-delivered {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .time-badge {
            font-size: 0.7rem;
            padding: 2px 8px;
            border-radius: 10px;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
        }
        
        .dispatch-btn {
            background: var(--primary);
            color: white;
            border: none;
            border-radius: 8px;
            padding: 8px 20px;
            font-weight: 500;
            transition: all 0.3s;
        }
        
        .dispatch-btn:hover {
            background: var(--primary-dark);
            transform: scale(1.05);
        }
        
        .track-btn {
            background: var(--info);
            color: white;
            border: none;
            border-radius: 8px;
            padding: 8px 20px;
            font-weight: 500;
        }
        
        .tab-content {
            background: white;
            border-radius: 0 0 12px 12px;
            padding: 20px;
            border: 1px solid #dee2e6;
            border-top: none;
        }
        
        .nav-tabs .nav-link {
            border: none;
            color: #6c757d;
            font-weight: 500;
            padding: 12px 24px;
        }
        
        .nav-tabs .nav-link.active {
            color: var(--primary);
            border-bottom: 3px solid var(--primary);
            background: transparent;
        }
        
        .order-actions {
            position: sticky;
            top: 20px;
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .count-badge {
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
            background: var(--primary);
            color: white;
        }
        
        .tracking-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .tracking-timeline {
            position: relative;
            padding-left: 30px;
            margin: 20px 0;
        }
        
        .tracking-timeline::before {
            content: '';
            position: absolute;
            left: 10px;
            top: 0;
            bottom: 0;
            width: 2px;
            background: var(--primary);
        }
        
        .tracking-step {
            position: relative;
            margin-bottom: 20px;
        }
        
        .tracking-step::before {
            content: '';
            position: absolute;
            left: -24px;
            top: 0;
            width: 14px;
            height: 14px;
            border-radius: 50%;
            background: white;
            border: 2px solid var(--primary);
        }
        
        .tracking-step.active::before {
            background: var(--primary);
        }
        
        .tracking-step.completed::before {
            background: var(--secondary);
            border-color: var(--secondary);
        }
        
        .rider-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            border: 1px solid #eaeaea;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .rider-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
        }
        
        .rider-card.selected {
            border: 2px solid var(--primary);
            background-color: rgba(211, 84, 0, 0.05);
        }
        
        .whatsapp-preview {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            white-space: pre-wrap;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .order-item {
            padding: 5px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .order-item:last-child {
            border-bottom: none;
        }
        
        .eta-indicator {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .eta-good {
            background: #d4edda;
            color: #155724;
        }
        
        .eta-warning {
            background: #fff3cd;
            color: #856404;
        }
        
        .eta-danger {
            background: #f8d7da;
            color: #721c24;
        }
        
        .tracking-code {
            font-family: monospace;
            font-size: 18px;
            font-weight: bold;
            letter-spacing: 2px;
            background: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            display: inline-block;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <?php 
    // Check if header.php exists
    $header_path = __DIR__ . '/header.php';
    if (file_exists($header_path)) {
        include $header_path;
    } else {
        echo '<nav class="navbar navbar-dark bg-dark">
            <div class="container-fluid">
                <a class="navbar-brand" href="#">
                    <i class="fas fa-truck"></i> Dispatch Management
                </a>
                <div class="d-flex">
                    <a href="reports.php" class="btn btn-outline-light me-2">
                        <i class="fas fa-chart-bar"></i> Reports
                    </a>
                    <a href="logout.php" class="btn btn-outline-light">
                        <i class="fas fa-sign-out-alt"></i> Logout
                    </a>
                </div>
            </div>
        </nav>';
    }
    ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-12">
                <!-- Outlet Selector -->
                <div class="mb-4">
                    <div class="btn-group" role="group">
                        <a href="?outlet=milano" 
                           class="btn <?php echo $outlet === 'milano' ? 'btn-warning' : 'btn-outline-warning'; ?>">
                            <i class="fas fa-pizza-slice"></i> Milano Pizza
                        </a>
                        <a href="?outlet=sushi" 
                           class="btn <?php echo $outlet === 'sushi' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-fish"></i> Sushi Bar
                        </a>
                    </div>
                </div>
                
                <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- Main Dashboard -->
                <div class="row">
                    <div class="col-md-8">
                        <!-- Paid Orders Section -->
                        <div class="card mb-4">
                            <div class="card-header bg-warning text-white">
                                <h5 class="mb-0">
                                    <i class="fas fa-clock me-2"></i>
                                    Paid Orders Ready for Dispatch
                                    <span class="badge bg-dark ms-2"><?php echo count($paid_orders); ?> orders</span>
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($paid_orders)): ?>
                                <div class="text-center py-5">
                                    <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                                    <h4>All orders have been dispatched!</h4>
                                    <p class="text-muted">No paid orders pending dispatch</p>
                                </div>
                                <?php else: ?>
                                <div class="row">
                                    <?php foreach ($paid_orders as $order): ?>
                                    <div class="col-md-6 mb-3">
                                        <div class="order-card paid">
                                            <div class="d-flex justify-content-between align-items-start mb-3">
                                                <div>
                                                    <h5 class="mb-1">Order #<?php echo $order['id']; ?></h5>
                                                    <small class="text-muted">
                                                        <i class="fas fa-clock me-1"></i>
                                                        <?php echo date('M j, H:i', strtotime($order['created_at'])); ?>
                                                        (<?php echo $order['minutes_old']; ?> min ago)
                                                    </small>
                                                </div>
                                                <span class="badge-status badge-paid">
                                                    <i class="fas fa-money-bill-wave me-1"></i>PAID
                                                </span>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <div class="fw-bold">
                                                    <i class="fas fa-user me-1"></i>
                                                    <?php echo htmlspecialchars($order['customer_name']); ?>
                                                </div>
                                                <div class="small text-muted">
                                                    <i class="fas fa-phone me-1"></i>
                                                    <?php echo htmlspecialchars($order['customer_phone']); ?>
                                                </div>
                                            </div>
                                            
                                            <?php if (!empty($order['delivery_location'])): ?>
                                            <div class="mb-3">
                                                <div class="small text-muted">
                                                    <i class="fas fa-map-marker-alt me-1"></i>
                                                    Delivery Address:
                                                </div>
                                                <div class="fw-bold"><?php echo htmlspecialchars($order['delivery_location']); ?></div>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <div class="mb-3">
                                                <div class="small text-muted">Order Items:</div>
                                                <div class="small"><?php 
                                                    if (!empty($order['items_summary'])) {
                                                        echo htmlspecialchars($order['items_summary']);
                                                    } else {
                                                        echo 'Order details available';
                                                    }
                                                ?></div>
                                            </div>
                                            
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div class="fw-bold text-success">
                                                    <i class="fas fa-money-bill-wave me-1"></i>
                                                    KES <?php echo number_format($order['total_amount'], 2); ?>
                                                </div>
                                                <button type="button" 
                                                        class="btn dispatch-btn"
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#dispatchModal"
                                                        onclick="setOrderForDispatch(<?php echo $order['id']; ?>, '<?php echo htmlspecialchars(addslashes($order['customer_name'])); ?>', '<?php echo htmlspecialchars(addslashes($order['customer_phone'])); ?>', <?php echo $order['total_amount']; ?>, '<?php echo htmlspecialchars(addslashes($order['delivery_location'] ?? '')); ?>')">
                                                    <i class="fas fa-paper-plane me-1"></i>Dispatch This Order
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Tracking Section -->
                        <div class="card">
                            <div class="card-header bg-info text-white">
                                <h5 class="mb-0">
                                    <i class="fas fa-shipping-fast me-2"></i>
                                    Order Tracking
                                    <span class="badge bg-dark ms-2"><?php echo count($dispatched_orders); ?> active</span>
                                </h5>
                            </div>
                            <div class="card-body">
                                <!-- Tracking Search -->
                                <div class="mb-4">
                                    <form method="GET" action="" class="row g-3">
                                        <input type="hidden" name="outlet" value="<?php echo $outlet; ?>">
                                        <div class="col-md-8">
                                            <div class="input-group">
                                                <span class="input-group-text">
                                                    <i class="fas fa-search"></i>
                                                </span>
                                                <input type="text" 
                                                       class="form-control" 
                                                       name="track" 
                                                       placeholder="Enter Order # or Tracking Code"
                                                       value="<?php echo htmlspecialchars($tracking_id); ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-search me-2"></i>Track Order
                                            </button>
                                        </div>
                                    </form>
                                </div>
                                
                                <?php if ($tracking_order): ?>
                                <!-- Single Order Tracking -->
                                <div class="tracking-card">
                                    <h5 class="text-white">
                                        <i class="fas fa-box me-2"></i>
                                        Order #<?php echo $tracking_order['id']; ?>
                                        <?php if ($tracking_order['tracking_number']): ?>
                                        <span class="tracking-code"><?php echo $tracking_order['tracking_number']; ?></span>
                                        <?php endif; ?>
                                    </h5>
                                    <p class="text-white-50 mb-3">
                                        Customer: <?php echo htmlspecialchars($tracking_order['customer_name']); ?>
                                    </p>
                                    
                                    <div class="tracking-timeline">
                                        <div class="tracking-step completed">
                                            <div class="fw-bold">Order Placed</div>
                                            <small class="text-white-50"><?php echo date('M j, H:i', strtotime($tracking_order['created_at'])); ?></small>
                                        </div>
                                        <div class="tracking-step completed">
                                            <div class="fw-bold">Payment Confirmed</div>
                                            <small class="text-white-50">Order paid and confirmed</small>
                                        </div>
                                        <div class="tracking-step completed">
                                            <div class="fw-bold">Order Dispatched</div>
                                            <small class="text-white-50">
                                                <?php echo $tracking_order['dispatch_time_formatted']; ?> 
                                                (<?php echo $tracking_order['minutes_since_dispatch']; ?> min ago)
                                            </small>
                                        </div>
                                        <div class="tracking-step <?php echo $tracking_order['minutes_remaining'] <= 0 ? 'active' : ''; ?>">
                                            <div class="fw-bold">Out for Delivery</div>
                                            <small class="text-white-50">
                                                ETA: <?php echo $tracking_order['eta_formatted']; ?>
                                                <?php 
                                                if ($tracking_order['minutes_remaining'] > 0) {
                                                    echo '(' . $tracking_order['minutes_remaining'] . ' min remaining)';
                                                    $eta_class = $tracking_order['minutes_remaining'] > 30 ? 'eta-good' : 
                                                                ($tracking_order['minutes_remaining'] > 10 ? 'eta-warning' : 'eta-danger');
                                                    echo '<span class="eta-indicator ' . $eta_class . ' ms-2">' . $tracking_order['minutes_remaining'] . ' min</span>';
                                                } else {
                                                    echo '<span class="eta-indicator eta-danger ms-2">OVERDUE</span>';
                                                }
                                                ?>
                                            </small>
                                        </div>
                                        <div class="tracking-step">
                                            <div class="fw-bold">Order Delivered</div>
                                            <small class="text-white-50">Awaiting delivery confirmation</small>
                                        </div>
                                    </div>
                                    
                                    <div class="row mt-4">
                                        <div class="col-md-6">
                                            <div class="bg-white bg-opacity-10 p-3 rounded">
                                                <small class="text-white-50">Rider Details</small>
                                                <div class="fw-bold"><?php echo htmlspecialchars($tracking_order['rider_name']); ?></div>
                                                <div class="small">
                                                    <i class="fas fa-phone me-1"></i>
                                                    <?php echo htmlspecialchars($tracking_order['rider_phone']); ?>
                                                </div>
                                                <div class="small">
                                                    <i class="fas fa-bicycle me-1"></i>
                                                    <?php echo htmlspecialchars($tracking_order['vehicle_type']); ?>
                                                    <?php if (!empty($tracking_order['vehicle_number'])): ?>
                                                    <span class="ms-2">
                                                        <i class="fas fa-hashtag me-1"></i>
                                                        <?php echo $tracking_order['vehicle_number']; ?>
                                                    </span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="bg-white bg-opacity-10 p-3 rounded">
                                                <small class="text-white-50">Delivery Address</small>
                                                <div class="fw-bold"><?php echo htmlspecialchars($tracking_order['delivery_location']); ?></div>
                                                <small class="text-white-50">
                                                    <i class="fas fa-clock me-1"></i>
                                                    Dispatch: <?php echo $tracking_order['dispatch_time_formatted']; ?>
                                                </small>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mt-4">
                                        <form method="POST" action="">
                                            <input type="hidden" name="action" value="mark_delivered">
                                            <input type="hidden" name="order_id" value="<?php echo $tracking_order['id']; ?>">
                                            <button type="submit" class="btn btn-success">
                                                <i class="fas fa-check-circle me-2"></i>Mark as Delivered
                                            </button>
                                            <a href="dispatch.php?outlet=<?php echo $outlet; ?>" class="btn btn-outline-light ms-2">
                                                Back to List
                                            </a>
                                        </form>
                                    </div>
                                </div>
                                
                                <?php elseif ($tracking_id): ?>
                                <!-- No order found -->
                                <div class="alert alert-warning">
                                    <i class="fas fa-exclamation-triangle me-2"></i>
                                    No order found with tracking code or order number: <?php echo htmlspecialchars($tracking_id); ?>
                                </div>
                                <?php endif; ?>
                                
                                <!-- Recent Dispatched Orders -->
                                <?php if (empty($tracking_id) && !empty($dispatched_orders)): ?>
                                <h6 class="mt-4 mb-3">Recently Dispatched Orders</h6>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Order #</th>
                                                <th>Customer</th>
                                                <th>Tracking</th>
                                                <th>Rider</th>
                                                <th>ETA</th>
                                                <th>Status</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($dispatched_orders as $order): ?>
                                            <tr>
                                                <td>
                                                    <strong>#<?php echo $order['id']; ?></strong>
                                                </td>
                                                <td><?php echo htmlspecialchars($order['customer_name']); ?></td>
                                                <td>
                                                    <code class="small"><?php echo $order['tracking_number'] ?? 'N/A'; ?></code>
                                                </td>
                                                <td><?php echo htmlspecialchars($order['rider_name'] ?? 'N/A'); ?></td>
                                                <td>
                                                    <?php echo $order['eta_formatted']; ?>
                                                    <?php if ($order['minutes_remaining'] > 0): ?>
                                                    <br>
                                                    <small class="<?php echo $order['minutes_remaining'] > 30 ? 'text-success' : 
                                                                ($order['minutes_remaining'] > 10 ? 'text-warning' : 'text-danger'); ?>">
                                                        <?php echo $order['minutes_remaining']; ?> min
                                                    </small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="badge-status badge-dispatched">
                                                        <i class="fas fa-truck me-1"></i>DISPATCHED
                                                    </span>
                                                </td>
                                                <td>
                                                    <a href="?outlet=<?php echo $outlet; ?>&track=<?php echo $order['id']; ?>" 
                                                       class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-eye me-1"></i>Track
                                                    </a>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <!-- Available Riders Sidebar -->
                        <div class="order-actions">
                            <h5 class="mb-3">
                                <i class="fas fa-motorcycle me-2 text-primary"></i>
                                Available Riders
                                <span class="count-badge ms-1"><?php echo count($available_riders); ?></span>
                            </h5>
                            
                            <?php if (empty($available_riders)): ?>
                            <div class="alert alert-warning">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                No riders available. Add riders below.
                            </div>
                            <?php else: ?>
                            <div id="riderList">
                                <?php foreach ($available_riders as $rider): ?>
                                <div class="rider-card" 
                                     id="rider_<?php echo $rider['id']; ?>"
                                     onclick="selectRider(<?php echo $rider['id']; ?>, '<?php echo htmlspecialchars(addslashes($rider['name'])); ?>', '<?php echo htmlspecialchars(addslashes($rider['phone'])); ?>', '<?php echo htmlspecialchars(addslashes($rider['vehicle_type'])); ?>', '<?php echo htmlspecialchars(addslashes($rider['vehicle_number'])); ?>')">
                                    <div class="d-flex justify-content-between align-items-start">
                                        <div>
                                            <h6 class="mb-1"><?php echo htmlspecialchars($rider['name']); ?></h6>
                                            <div class="small mb-2">
                                                <i class="fas fa-phone text-muted me-1"></i>
                                                <?php echo htmlspecialchars($rider['phone']); ?>
                                            </div>
                                            <div class="small">
                                                <i class="fas fa-bicycle text-muted me-1"></i>
                                                <?php echo htmlspecialchars($rider['vehicle_type']); ?>
                                                <?php if (!empty($rider['vehicle_number'])): ?>
                                                <span class="ms-2">
                                                    <i class="fas fa-hashtag text-muted me-1"></i>
                                                    <?php echo htmlspecialchars($rider['vehicle_number']); ?>
                                                </span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <div class="text-end">
                                            <span class="badge bg-success">Available</span>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Add Rider Form -->
                            <div class="mt-4 pt-3 border-top">
                                <h6 class="mb-3">Add New Rider</h6>
                                <form method="POST" action="">
                                    <input type="hidden" name="action" value="add_rider">
                                    <div class="mb-3">
                                        <input type="text" class="form-control" name="name" required placeholder="Rider Name">
                                    </div>
                                    <div class="mb-3">
                                        <input type="tel" class="form-control" name="phone" required placeholder="Phone Number">
                                    </div>
                                    <div class="mb-3">
                                        <select class="form-select" name="vehicle_type">
                                            <option value="Motorcycle">Motorcycle</option>
                                            <option value="Bicycle">Bicycle</option>
                                            <option value="Car">Car</option>
                                            <option value="Scooter">Scooter</option>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <input type="text" class="form-control" name="vehicle_number" placeholder="Vehicle Number (Optional)">
                                    </div>
                                    <button type="submit" class="btn btn-primary w-100">
                                        <i class="fas fa-plus me-2"></i>Add Rider
                                    </button>
                                </form>
                            </div>
                            
                            <!-- Quick Stats -->
                            <div class="mt-4 pt-3 border-top">
                                <h6 class="mb-3">Quick Stats</h6>
                                <div class="row text-center">
                                    <div class="col-6 mb-3">
                                        <div class="display-6 text-warning"><?php echo count($paid_orders); ?></div>
                                        <small class="text-muted">Pending</small>
                                    </div>
                                    <div class="col-6 mb-3">
                                        <div class="display-6 text-success"><?php echo count($dispatched_orders); ?></div>
                                        <small class="text-muted">Active</small>
                                    </div>
                                    <div class="col-6">
                                        <div class="display-6 text-primary"><?php echo count($available_riders); ?></div>
                                        <small class="text-muted">Riders</small>
                                    </div>
                                    <div class="col-6">
                                        <div class="display-6 text-info">
                                            <?php 
                                            $total = count($paid_orders) + count($dispatched_orders);
                                            echo $total > 99 ? '99+' : $total;
                                            ?>
                                        </div>
                                        <small class="text-muted">Total</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Dispatch Modal -->
    <div class="modal fade" id="dispatchModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <form method="POST" action="">
                    <input type="hidden" name="action" value="dispatch">
                    <input type="hidden" name="order_id" id="modalOrderId">
                    
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="fas fa-paper-plane me-2"></i>
                            Dispatch Order
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    
                    <div class="modal-body">
                        <!-- Order Details -->
                        <div class="card mb-3">
                            <div class="card-body">
                                <h6>Order Details</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <small class="text-muted">Order #</small>
                                        <div id="modalOrderNumber" class="fw-bold"></div>
                                    </div>
                                    <div class="col-md-6">
                                        <small class="text-muted">Customer</small>
                                        <div id="modalCustomerName" class="fw-bold"></div>
                                    </div>
                                    <div class="col-md-6">
                                        <small class="text-muted">Contact</small>
                                        <div id="modalCustomerPhone"></div>
                                    </div>
                                    <div class="col-md-6">
                                        <small class="text-muted">Amount</small>
                                        <div id="modalOrderAmount" class="text-success fw-bold"></div>
                                    </div>
                                    <div class="col-md-12 mt-2">
                                        <small class="text-muted">Delivery Address</small>
                                        <div id="modalDeliveryAddress" class="small"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Select Rider -->
                        <div class="mb-4">
                            <h6>Select Rider</h6>
                            <?php if (empty($available_riders)): ?>
                            <div class="alert alert-warning">
                                No riders available. Please add riders first.
                            </div>
                            <?php else: ?>
                            <div class="row" id="riderSelection">
                                <?php foreach ($available_riders as $rider): ?>
                                <div class="col-md-6 mb-3">
                                    <div class="rider-card" 
                                         onclick="selectModalRider(<?php echo $rider['id']; ?>)">
                                        <div class="form-check">
                                            <input class="form-check-input" 
                                                   type="radio" 
                                                   name="rider_id" 
                                                   value="<?php echo $rider['id']; ?>"
                                                   id="rider_option_<?php echo $rider['id']; ?>"
                                                   required>
                                            <label class="form-check-label" for="rider_option_<?php echo $rider['id']; ?>">
                                                <strong><?php echo htmlspecialchars($rider['name']); ?></strong>
                                            </label>
                                        </div>
                                        <div class="small mt-2">
                                            <i class="fas fa-phone text-muted me-1"></i>
                                            <?php echo htmlspecialchars($rider['phone']); ?>
                                        </div>
                                        <div class="small">
                                            <i class="fas fa-bicycle text-muted me-1"></i>
                                            <?php echo htmlspecialchars($rider['vehicle_type']); ?>
                                            <?php if (!empty($rider['vehicle_number'])): ?>
                                            <span class="ms-2">
                                                <i class="fas fa-hashtag text-muted me-1"></i>
                                                <?php echo htmlspecialchars($rider['vehicle_number']); ?>
                                            </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- WhatsApp Message Preview -->
                        <div class="mb-3">
                            <h6>WhatsApp Message Preview</h6>
                            <div class="whatsapp-preview" id="whatsappPreview">
                                Select a rider to preview message...
                            </div>
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary" id="confirmDispatchBtn" <?php echo empty($available_riders) ? 'disabled' : ''; ?>>
                            <i class="fas fa-paper-plane me-2"></i>
                            Confirm Dispatch & Send WhatsApp
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- JavaScript -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Global variables
        let currentOrder = {};
        let selectedRiderId = null;
        let selectedRiderDetails = {};
        
        function setOrderForDispatch(orderId, customerName, customerPhone, amount, deliveryAddress) {
            currentOrder = {
                id: orderId,
                customerName: customerName,
                customerPhone: customerPhone,
                amount: amount,
                deliveryAddress: deliveryAddress || 'Pickup'
            };
            
            document.getElementById('modalOrderId').value = orderId;
            document.getElementById('modalOrderNumber').textContent = '#' + orderId;
            document.getElementById('modalCustomerName').textContent = customerName;
            document.getElementById('modalCustomerPhone').textContent = customerPhone;
            document.getElementById('modalOrderAmount').textContent = 'KES ' + parseFloat(amount).toFixed(2);
            document.getElementById('modalDeliveryAddress').textContent = deliveryAddress || 'Pickup';
            
            // Reset rider selection
            document.querySelectorAll('input[name="rider_id"]').forEach(radio => {
                radio.checked = false;
            });
            document.querySelectorAll('.rider-card').forEach(card => {
                card.classList.remove('selected');
            });
            document.getElementById('confirmDispatchBtn').disabled = true;
            document.getElementById('whatsappPreview').innerHTML = 'Select a rider to preview message...';
            
            selectedRiderId = null;
            selectedRiderDetails = {};
        }
        
        function selectModalRider(riderId) {
            selectedRiderId = riderId;
            
            // Update UI
            document.querySelectorAll('.rider-card').forEach(card => {
                card.classList.remove('selected');
            });
            event.currentTarget.classList.add('selected');
            document.getElementById('rider_option_' + riderId).checked = true;
            
            // Enable dispatch button
            document.getElementById('confirmDispatchBtn').disabled = false;
            
            // Get rider details from the clicked card
            const riderCard = event.currentTarget;
            const riderName = riderCard.querySelector('strong').textContent;
            const riderPhone = riderCard.querySelector('.fa-phone').parentNode.textContent.trim();
            const riderVehicle = riderCard.querySelector('.fa-bicycle').parentNode.textContent.trim();
            
            selectedRiderDetails = {
                name: riderName,
                phone: riderPhone,
                vehicle: riderVehicle
            };
            
            updateWhatsAppPreview();
        }
        
        function selectRider(riderId, riderName, riderPhone, riderVehicle, riderNumber) {
            // For sidebar selection (optional feature)
            selectedRiderId = riderId;
            selectedRiderDetails = {
                name: riderName,
                phone: riderPhone,
                vehicle: riderVehicle + (riderNumber ? ' (' + riderNumber + ')' : '')
            };
            
            // Highlight selected rider in sidebar
            document.querySelectorAll('.rider-card').forEach(card => {
                card.classList.remove('selected');
            });
            document.getElementById('rider_' + riderId).classList.add('selected');
            
            console.log('Selected rider:', selectedRiderDetails);
        }
        
        function updateWhatsAppPreview() {
            if (!selectedRiderId || !currentOrder.id) return;
            
            const eta = Math.floor(Math.random() * 20) + 20; // 20-40 minutes
            const etaTime = new Date(Date.now() + eta * 60000);
            const etaStr = etaTime.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            const trackingNumber = 'TRK' + String(currentOrder.id).padStart(6, '0') + Math.random().toString(36).substr(2, 6).toUpperCase();
            
            const preview = document.getElementById('whatsappPreview');
            preview.innerHTML = `📦 ORDER DISPATCHED
Order #${currentOrder.id}
Tracking: ${trackingNumber}
────────────────────────────
👤 Customer: ${currentOrder.customerName}
📍 Delivery to: ${currentOrder.deliveryAddress}

💰 Total: KES ${parseFloat(currentOrder.amount).toFixed(2)}

🚀 Dispatch Details:
🏍️ Rider: ${selectedRiderDetails.name}
📞 Rider Contact: ${selectedRiderDetails.phone}
🚗 Vehicle: ${selectedRiderDetails.vehicle}
⏰ ETA: ${etaStr} (approx. ${eta} minutes)

📱 Track your order:
Link: https://yourdomain.com/track.php?code=${trackingNumber}

Thank you for choosing ${currentOrder.deliveryAddress.includes('Pizza') ? 'Milano Pizza' : 'Sushi Bar'}! 🎉
For inquiries, call: +254700123456`;
        }
        
        // Auto-refresh orders every 30 seconds
        setInterval(() => {
            if (!document.hidden) {
                const paidCount = document.querySelectorAll('.order-card.paid').length;
                if (paidCount > 0) {
                    // Only refresh if there are pending orders
                    location.reload();
                }
            }
        }, 30000);
        
        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            // Add click handlers to all rider cards in modal
            document.querySelectorAll('.rider-card').forEach(card => {
                card.addEventListener('click', function() {
                    const radio = this.querySelector('input[type="radio"]');
                    if (radio) {
                        const riderId = radio.value;
                        selectModalRider(riderId);
                    }
                });
            });
            
            // Show auto-refresh notification
            setTimeout(() => {
                const refreshAlert = document.createElement('div');
                refreshAlert.className = 'alert alert-info alert-dismissible fade show mt-3';
                refreshAlert.innerHTML = `
                    <i class="fas fa-sync-alt me-2"></i>
                    Orders auto-refresh every 30 seconds
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.querySelector('.container-fluid').prepend(refreshAlert);
            }, 5000);
        });
    </script>
</body>
</html>