<?php
// admin/index.php - DUAL OUTLET Admin Dashboard

session_start();

// Check if admin is logged in
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

// Check if outlet parameter is set
$outlet = isset($_GET['outlet']) ? strtolower(trim($_GET['outlet'])) : 'sushi';
if (!in_array($outlet, ['sushi', 'milano'])) {
    $outlet = 'sushi';
}

// Get statistics for specific outlet
$stats = [];

// Total orders for outlet
$stmt = $conn->prepare("SELECT COUNT(*) as total FROM orders WHERE outlet = :outlet");
$stmt->execute([':outlet' => $outlet]);
$stats['total_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

// Today's orders for outlet
$stmt = $conn->prepare("SELECT COUNT(*) as total FROM orders WHERE outlet = :outlet AND DATE(created_at) = CURDATE()");
$stmt->execute([':outlet' => $outlet]);
$stats['today_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

// Pending orders for outlet
$stmt = $conn->prepare("SELECT COUNT(*) as total FROM orders WHERE outlet = :outlet AND (status = 'pending' OR status = 'processing')");
$stmt->execute([':outlet' => $outlet]);
$stats['pending_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

// Total revenue for outlet
$stmt = $conn->prepare("SELECT SUM(total_amount) as total FROM orders WHERE outlet = :outlet AND (status = 'completed' OR status = 'paid')");
$stmt->execute([':outlet' => $outlet]);
$stats['total_revenue'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

// Today's revenue for outlet
$stmt = $conn->prepare("SELECT SUM(total_amount) as total FROM orders WHERE outlet = :outlet AND DATE(created_at) = CURDATE() AND (status = 'completed' OR status = 'paid')");
$stmt->execute([':outlet' => $outlet]);
$stats['today_revenue'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

// Get recent orders for outlet
$stmt = $conn->prepare("SELECT * FROM orders WHERE outlet = :outlet ORDER BY created_at DESC LIMIT 10");
$stmt->execute([':outlet' => $outlet]);
$recent_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get popular items for outlet
$stmt = $conn->prepare("
    SELECT oi.item_name, SUM(oi.quantity) as total_quantity 
    FROM order_items oi
    JOIN orders o ON oi.order_id = o.id
    WHERE o.outlet = :outlet
    GROUP BY oi.item_name 
    ORDER BY total_quantity DESC 
    LIMIT 5
");
$stmt->execute([':outlet' => $outlet]);
$popular_items = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get delivery locations stats for outlet
$stmt = $conn->prepare("
    SELECT delivery_location, COUNT(*) as order_count 
    FROM orders 
    WHERE outlet = :outlet
    GROUP BY delivery_location 
    ORDER BY order_count DESC 
    LIMIT 5
");
$stmt->execute([':outlet' => $outlet]);
$delivery_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get total stats for both outlets
$stmt = $conn->query("SELECT outlet, COUNT(*) as total_orders, SUM(total_amount) as total_revenue FROM orders GROUP BY outlet");
$outlet_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo strtoupper($outlet); ?> Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        :root {
            --sushi-primary: #e53935;
            --sushi-primary-dark: #c62828;
            --milano-primary: #D35400;
            --milano-primary-dark: #A04000;
            --secondary: #2e7d32;
            --light: #f5f5f5;
            --dark: #333;
            --gray: #777;
        }
        
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .sidebar {
            background: linear-gradient(180deg, var(--sushi-primary) 0%, var(--sushi-primary-dark) 100%);
            color: white;
            min-height: 100vh;
            position: fixed;
            width: 280px;
            transition: all 0.3s;
            z-index: 1000;
        }
        
        .sidebar.milano {
            background: linear-gradient(180deg, var(--milano-primary) 0%, var(--milano-primary-dark) 100%);
        }
        
        .sidebar-header {
            padding: 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .sidebar-header h3 {
            margin: 0;
            font-weight: 700;
            color: white;
        }
        
        .sidebar-header .outlet-switcher {
            margin-top: 10px;
        }
        
        .outlet-switch-btn {
            background: rgba(255,255,255,0.1);
            border: 1px solid rgba(255,255,255,0.2);
            color: white;
            padding: 5px 10px;
            border-radius: 5px;
            text-decoration: none;
            font-size: 0.9rem;
            display: inline-block;
            margin: 2px;
            transition: all 0.3s;
        }
        
        .outlet-switch-btn:hover {
            background: rgba(255,255,255,0.2);
            color: white;
        }
        
        .outlet-switch-btn.active {
            background: white;
            color: var(--sushi-primary);
        }
        
        .sidebar.milano .outlet-switch-btn.active {
            color: var(--milano-primary);
        }
        
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 5px 10px;
            border-radius: 8px;
            transition: all 0.3s;
        }
        
        .sidebar .nav-link:hover,
        .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
        }
        
        .sidebar .nav-link i {
            width: 25px;
            text-align: center;
        }
        
        .main-content {
            margin-left: 280px;
            padding: 20px;
            transition: all 0.3s;
        }
        
        .navbar {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 15px 20px;
            margin-bottom: 20px;
            border-radius: 10px;
        }
        
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 15px rgba(0,0,0,0.08);
            transition: transform 0.3s;
            border-left: 4px solid var(--sushi-primary);
        }
        
        .stat-card.milano {
            border-left-color: var(--milano-primary);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-card .stat-icon {
            font-size: 2.5rem;
            color: var(--sushi-primary);
            margin-bottom: 15px;
        }
        
        .stat-card.milano .stat-icon {
            color: var(--milano-primary);
        }
        
        .stat-card .stat-number {
            font-size: 2rem;
            font-weight: 700;
            color: var(--dark);
        }
        
        .stat-card .stat-label {
            color: var(--gray);
            font-size: 0.9rem;
        }
        
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 3px 15px rgba(0,0,0,0.08);
            margin-bottom: 20px;
        }
        
        .card-header {
            background: white;
            border-bottom: 1px solid #eee;
            padding: 15px 20px;
            font-weight: 600;
            color: var(--dark);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .table th {
            border-top: none;
            font-weight: 600;
            color: var(--gray);
            background: #f8f9fa;
        }
        
        .badge-pending {
            background: #ffc107;
            color: #212529;
        }
        
        .badge-processing {
            background: #17a2b8;
            color: white;
        }
        
        .badge-completed {
            background: var(--secondary);
            color: white;
        }
        
        .badge-paid {
            background: #28a745;
            color: white;
        }
        
        .badge-cancelled {
            background: #dc3545;
            color: white;
        }
        
        .badge-outlet {
            background: #6c757d;
            color: white;
            font-size: 0.7rem;
            padding: 3px 8px;
        }
        
        .badge-outlet.sushi {
            background: var(--sushi-primary);
        }
        
        .badge-outlet.milano {
            background: var(--milano-primary);
        }
        
        .btn-primary {
            background: var(--sushi-primary);
            border: none;
            padding: 8px 20px;
            border-radius: 5px;
        }
        
        .btn-primary:hover {
            background: var(--sushi-primary-dark);
        }
        
        .btn-outlet-milano {
            background: var(--milano-primary);
            border: none;
            color: white;
            padding: 8px 20px;
            border-radius: 5px;
        }
        
        .btn-outlet-milano:hover {
            background: var(--milano-primary-dark);
            color: white;
        }
        
        .btn-outline-primary {
            color: var(--sushi-primary);
            border-color: var(--sushi-primary);
        }
        
        .btn-outline-primary:hover {
            background: var(--sushi-primary);
            border-color: var(--sushi-primary);
        }
        
        .outlet-summary-card {
            border-left: 4px solid;
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
            background: white;
        }
        
        .outlet-summary-card.sushi {
            border-left-color: var(--sushi-primary);
        }
        
        .outlet-summary-card.milano {
            border-left-color: var(--milano-primary);
        }
        
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -280px;
                width: 280px;
            }
            
            .sidebar.active {
                margin-left: 0;
            }
            
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar <?php echo $outlet === 'milano' ? 'milano' : ''; ?>" id="sidebar">
        <div class="sidebar-header">
            <h3>
                <?php if ($outlet === 'sushi'): ?>
                    SUSHI<span>-BAR</span> Admin
                <?php else: ?>
                    MILANO<span> PIZZA-PASTA</span> Admin
                <?php endif; ?>
            </h3>
            <small>Restaurant Management System</small>
            
            <div class="outlet-switcher">
                <a href="?outlet=sushi" class="outlet-switch-btn <?php echo $outlet === 'sushi' ? 'active' : ''; ?>">
                    <i class="fas fa-fish"></i> Sushi-Bar
                </a>
                <a href="?outlet=milano" class="outlet-switch-btn <?php echo $outlet === 'milano' ? 'active' : ''; ?>">
                    <i class="fas fa-pizza-slice"></i> Milano Pizza-Pasta
                </a>
            </div>
        </div>
        
        <nav class="nav flex-column mt-3">
            <a class="nav-link active" href="index.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-tachometer-alt"></i> Dashboard
            </a>
            <a class="nav-link" href="orders.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-shopping-cart"></i> Orders
                <span class="badge bg-danger float-end"><?php echo $stats['pending_orders']; ?></span>
            </a>
            <a class="nav-link" href="menu.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-utensils"></i> Menu Management
            </a>
            <a class="nav-link" href="customers.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-users"></i> Customers
            </a>
            <a class="nav-link" href="reports.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-chart-bar"></i> Reports
            </a>
            <a class="nav-link" href="delivery.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-map-marker-alt"></i> Delivery Areas
            </a>
            <a class="nav-link" href="mpesa_transactions.php?outlet=<?php echo $outlet; ?>">
                <i class="fas fa-money-bill-wave"></i> M-Pesa Transactions
            </a>
            <div class="mt-4"></div>
            <!-- Combined Dashboard Link (Existing) -->
            <a class="nav-link" href="combined_dashboard.php">
                <i class="fas fa-store"></i> All Outlets View
            </a>
            <a class="nav-link" href="settings.php">
                <i class="fas fa-cog"></i> Settings
            </a>
            <a class="nav-link" href="logout.php">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </nav>
    </div>
    
    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Navbar -->
        <nav class="navbar">
            <div class="container-fluid">
                <button class="navbar-toggler d-md-none" type="button" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <div class="d-none d-md-block">
                    <h4 class="mb-0">
                        <?php if ($outlet === 'sushi'): ?>
                            SUSHI-BAR Dashboard
                        <?php else: ?>
                            MILANO PIZZA-PASTA Dashboard
                        <?php endif; ?>
                        <span class="badge badge-outlet <?php echo $outlet; ?> ms-2">
                            <?php echo strtoupper($outlet); ?> OUTLET
                        </span>
                    </h4>
                </div>
                <div class="d-flex align-items-center">
                    <span class="me-3">
                        <i class="fas fa-calendar-day"></i> <?php echo date('F j, Y'); ?>
                    </span>
                    <!-- ADDED: Combined Dashboard Button in Navbar -->
                    <a href="combined_dashboard.php" class="btn btn-outline-primary btn-sm me-2">
                        <i class="fas fa-store me-1"></i> All Outlets
                    </a>
                    <div class="dropdown">
                        <button class="btn btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user-circle"></i> Admin
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="profile.php"><i class="fas fa-user"></i> Profile</a></li>
                            <li><a class="dropdown-item" href="settings.php"><i class="fas fa-cog"></i> Settings</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
                        </ul>
                    </div>
                </div>
            </div>
        </nav>
        
        <!-- Outlet Summary Row -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="outlet-summary-card sushi">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="mb-1"><i class="fas fa-fish me-2"></i>SUSHI-BAR</h6>
                            <?php 
                            $sushi_stats = array_filter($outlet_stats, fn($s) => $s['outlet'] === 'sushi');
                            $sushi_stat = reset($sushi_stats);
                            ?>
                            <div class="text-muted small">
                                <?php echo $sushi_stat['total_orders'] ?? 0; ?> orders | 
                                KES <?php echo number_format($sushi_stat['total_revenue'] ?? 0, 2); ?> revenue
                            </div>
                        </div>
                        <a href="?outlet=sushi" class="btn btn-primary btn-sm">Go to Dashboard</a>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="outlet-summary-card milano">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="mb-1"><i class="fas fa-pizza-slice me-2"></i>MILANO PIZZA-PASTA</h6>
                            <?php 
                            $milano_stats = array_filter($outlet_stats, fn($s) => $s['outlet'] === 'milano');
                            $milano_stat = reset($milano_stats);
                            ?>
                            <div class="text-muted small">
                                <?php echo $milano_stat['total_orders'] ?? 0; ?> orders | 
                                KES <?php echo number_format($milano_stat['total_revenue'] ?? 0, 2); ?> revenue
                            </div>
                        </div>
                        <a href="?outlet=milano" class="btn btn-outlet-milano btn-sm">Go to Dashboard</a>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Statistics Cards -->
        <div class="row">
            <div class="col-md-3">
                <div class="stat-card <?php echo $outlet === 'milano' ? 'milano' : ''; ?>">
                    <div class="stat-icon">
                        <i class="fas fa-shopping-cart"></i>
                    </div>
                    <div class="stat-number"><?php echo number_format($stats['total_orders']); ?></div>
                    <div class="stat-label">Total Orders</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card <?php echo $outlet === 'milano' ? 'milano' : ''; ?>">
                    <div class="stat-icon">
                        <i class="fas fa-calendar-day"></i>
                    </div>
                    <div class="stat-number"><?php echo number_format($stats['today_orders']); ?></div>
                    <div class="stat-label">Today's Orders</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card <?php echo $outlet === 'milano' ? 'milano' : ''; ?>">
                    <div class="stat-icon">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-number"><?php echo number_format($stats['pending_orders']); ?></div>
                    <div class="stat-label">Pending Orders</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card <?php echo $outlet === 'milano' ? 'milano' : ''; ?>">
                    <div class="stat-icon">
                        <i class="fas fa-money-bill-wave"></i>
                    </div>
                    <div class="stat-number">KES <?php echo number_format($stats['total_revenue'], 2); ?></div>
                    <div class="stat-label">Total Revenue</div>
                </div>
            </div>
        </div>
        
        <!-- Main Content Row -->
        <div class="row">
            <!-- Recent Orders -->
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span>Recent Orders (<?php echo strtoupper($outlet); ?>)</span>
                        <a href="orders.php?outlet=<?php echo $outlet; ?>" class="btn <?php echo $outlet === 'milano' ? 'btn-outlet-milano' : 'btn-primary'; ?> btn-sm">View All</a>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Order #</th>
                                        <th>Customer</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_orders as $order): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($order['order_number']); ?></strong>
                                            <br>
                                            <small class="badge badge-outlet <?php echo $order['outlet'] ?? 'sushi'; ?>">
                                                <?php echo strtoupper($order['outlet'] ?? 'sushi'); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($order['customer_name']); ?></div>
                                            <small class="text-muted"><?php echo htmlspecialchars($order['customer_phone']); ?></small>
                                        </td>
                                        <td>KES <?php echo number_format($order['total_amount'], 2); ?></td>
                                        <td>
                                            <?php
                                            $status_class = '';
                                            switch($order['status']) {
                                                case 'pending': $status_class = 'badge-pending'; break;
                                                case 'processing': $status_class = 'badge-processing'; break;
                                                case 'completed': $status_class = 'badge-completed'; break;
                                                case 'paid': $status_class = 'badge-paid'; break;
                                                case 'cancelled': $status_class = 'badge-cancelled'; break;
                                                default: $status_class = 'badge-pending';
                                            }
                                            ?>
                                            <span class="badge <?php echo $status_class; ?>">
                                                <?php echo ucfirst($order['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M j, g:i A', strtotime($order['created_at'])); ?></td>
                                        <td>
                                            <a href="order_details.php?id=<?php echo $order['id']; ?>&outlet=<?php echo $outlet; ?>" 
                                               class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    <?php if (empty($recent_orders)): ?>
                                    <tr>
                                        <td colspan="6" class="text-center text-muted py-4">
                                            <i class="fas fa-shopping-cart fa-2x mb-3"></i>
                                            <p>No orders found for this outlet.</p>
                                        </td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Sidebar Stats -->
            <div class="col-md-4">
                <!-- Popular Items -->
                <div class="card">
                    <div class="card-header">
                        <span>Popular Items (<?php echo strtoupper($outlet); ?>)</span>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($popular_items)): ?>
                        <ul class="list-group list-group-flush">
                            <?php foreach ($popular_items as $item): ?>
                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                <?php echo htmlspecialchars($item['item_name']); ?>
                                <span class="badge <?php echo $outlet === 'milano' ? 'bg-warning' : 'bg-primary'; ?> rounded-pill">
                                    <?php echo $item['total_quantity']; ?>
                                </span>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                        <?php else: ?>
                        <div class="text-center text-muted py-4">
                            <i class="fas fa-utensils fa-2x mb-3"></i>
                            <p>No popular items yet.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Delivery Locations -->
                <div class="card mt-3">
                    <div class="card-header">
                        <span>Top Delivery Locations (<?php echo strtoupper($outlet); ?>)</span>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($delivery_stats)): ?>
                        <ul class="list-group list-group-flush">
                            <?php foreach ($delivery_stats as $location): ?>
                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                <?php echo htmlspecialchars($location['delivery_location']); ?>
                                <span class="badge bg-success rounded-pill"><?php echo $location['order_count']; ?></span>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                        <?php else: ?>
                        <div class="text-center text-muted py-4">
                            <i class="fas fa-map-marker-alt fa-2x mb-3"></i>
                            <p>No delivery data yet.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="card mt-3">
                    <div class="card-header">
                        <span>Quick Actions</span>
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <a href="new_order.php?outlet=<?php echo $outlet; ?>" 
                               class="btn <?php echo $outlet === 'milano' ? 'btn-outlet-milano' : 'btn-primary'; ?>">
                                <i class="fas fa-plus"></i> Create New Order
                            </a>
                            <a href="reports.php?period=today&outlet=<?php echo $outlet; ?>" class="btn btn-outline-primary">
                                <i class="fas fa-chart-line"></i> Today's Report
                            </a>
                            <a href="mpesa_transactions.php?outlet=<?php echo $outlet; ?>" class="btn btn-outline-primary">
                                <i class="fas fa-money-check"></i> Check Payments
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle sidebar on mobile
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });
        
        // Auto-refresh dashboard every 30 seconds
        setInterval(function() {
            const outlet = '<?php echo $outlet; ?>';
            fetch('dashboard_stats.php?outlet=' + outlet)
                .then(response => response.json())
                .then(data => {
                    // Update stats
                    document.querySelectorAll('.stat-card .stat-number').forEach((el, index) => {
                        if (index === 0) el.textContent = data.total_orders.toLocaleString();
                        if (index === 1) el.textContent = data.today_orders.toLocaleString();
                        if (index === 2) el.textContent = data.pending_orders.toLocaleString();
                        if (index === 3) el.textContent = 'KES ' + data.total_revenue.toLocaleString('en-US', {minimumFractionDigits: 2});
                    });
                    
                    // Update pending badge
                    const badge = document.querySelector('.sidebar .badge');
                    if (badge) {
                        badge.textContent = data.pending_orders;
                    }
                })
                .catch(error => console.error('Error updating stats:', error));
        }, 30000);
    </script>
</body>
</html>