<?php
session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$db   = new Database();
$conn = $db->getConnection();

$outlet = $_GET['outlet'] ?? 'milano';
$action = $_GET['action'] ?? 'list';
$id = $_GET['id'] ?? null;
$itemType = $_GET['type'] ?? null; // 'pizza' or 'regular'

// Handle old action names for backward compatibility
if ($action === 'add_item') {
    header("Location: menu.php?outlet=$outlet&action=add&type=regular");
    exit;
}

if ($action === 'edit_item' && $id) {
    // Check what type of item it is
    try {
        $stmt = $conn->prepare("SELECT category FROM menu_items WHERE id = ? AND outlet = ?");
        $stmt->execute([$id, $outlet]);
        $item = $stmt->fetch(PDO::FETCH_ASSOC);
        $type = ($item && $item['category'] === 'pizza') ? 'pizza' : 'regular';
        header("Location: menu.php?outlet=$outlet&action=edit&id=$id&type=$type");
        exit;
    } catch (Exception $e) {
        // Fallback
        header("Location: menu.php?outlet=$outlet&action=edit&id=$id&type=regular");
        exit;
    }
}

// Also handle the old pizza actions for backward compatibility
if ($action === 'add_pizza') {
    header("Location: menu.php?outlet=$outlet&action=add&type=pizza");
    exit;
}

if ($action === 'edit_pizza' && $id) {
    header("Location: menu.php?outlet=$outlet&action=edit&id=$id&type=pizza");
    exit;
}

// Handle different actions
if ($action === 'add' || $action === 'edit') {
    
    // Determine if this is a pizza action
    if ($action === 'add') {
        // For adding, use the type parameter
        $isPizza = ($itemType === 'pizza');
    } else {
        // For editing, check the actual item from database
        if (!$id) {
            $_SESSION['error_message'] = "Item ID required for editing";
            header("Location: menu.php?outlet=$outlet");
            exit;
        }
        
        try {
            $stmt = $conn->prepare("SELECT * FROM menu_items WHERE id = ? AND outlet = ?");
            $stmt->execute([$id, $outlet]);
            $item = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$item) {
                $_SESSION['error_message'] = "Menu item not found";
                header("Location: menu.php?outlet=$outlet");
                exit;
            }
            
            $isPizza = ($item['category'] === 'pizza');
            
            // If type parameter was provided and doesn't match, redirect
            if ($itemType && (($itemType === 'pizza' && !$isPizza) || ($itemType !== 'pizza' && $isPizza))) {
                $correctType = $isPizza ? 'pizza' : 'regular';
                header("Location: menu.php?outlet=$outlet&action=edit&id=$id&type=$correctType");
                exit;
            }
            
        } catch (Exception $e) {
            $_SESSION['error_message'] = "Error loading item: " . $e->getMessage();
            header("Location: menu.php?outlet=$outlet");
            exit;
        }
    }
    
    // Load pizza categories only for pizza forms
    if ($isPizza) {
        $pizzaCategories = [];
        try {
            $catStmt = $conn->query("SELECT * FROM pizza_categories WHERE is_active = 1 ORDER BY display_order");
            if ($catStmt) {
                $pizzaCategories = $catStmt->fetchAll(PDO::FETCH_ASSOC);
            }
        } catch (Exception $e) {
            // Default pizza categories if table doesn't exist
            $pizzaCategories = [
                ['id' => 1, 'name' => 'Classic Pizzas', 'slug' => 'classic'],
                ['id' => 2, 'name' => 'Premium Pizzas', 'slug' => 'premium'],
                ['id' => 3, 'name' => 'Special Pizzas', 'slug' => 'special']
            ];
        }
        
        // Default size prices for pizza
        $defaultSizes = [
            'small' => 0,
            'medium' => 0,
            'large' => 0,
            'giant' => 0
        ];
        
        // Parse sizes if editing pizza
        if ($id && !empty($item['sizes'])) {
            $sizesData = json_decode($item['sizes'], true);
            if ($sizesData) {
                foreach ($defaultSizes as $size => $value) {
                    if (isset($sizesData[$size])) {
                        $defaultSizes[$size] = $sizesData[$size];
                    }
                }
            }
        }
    }
    
    // Get available categories for non-pizza items
    $categories = [];
    if (!$isPizza) {
        // Load categories from database or use defaults
        try {
            $catStmt = $conn->query("SELECT DISTINCT category FROM menu_items WHERE outlet='$outlet' AND category != 'pizza' ORDER BY category");
            if ($catStmt) {
                $categories = $catStmt->fetchAll(PDO::FETCH_COLUMN, 0);
            }
        } catch (Exception $e) {
            // Continue with empty categories
        }
        
        // Add common categories if none exist
        if (empty($categories)) {
            $categories = ['appetizer', 'main course', 'dessert', 'beverage', 'salad', 'soup'];
        }
        
        // Remove 'pizza' from categories if it exists
        $categories = array_filter($categories, function($cat) {
            return strtolower($cat) !== 'pizza';
        });
        
        // Add "other" option
        $categories[] = 'other';
    }
    
    // Set default item values for new items
    if ($action === 'add') {
        $item = [
            'id' => null,
            'name' => '',
            'description' => '',
            'category' => $isPizza ? 'pizza' : ($categories[0] ?? 'appetizer'),
            'price' => '',
            'sizes' => '',
            'image_url' => '',
            'is_active' => 1,
            'pizza_category_id' => null
        ];
    }
    
    // Display the form
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title><?= $id ? 'Edit' : 'Add' ?> <?= $isPizza ? 'Pizza' : 'Item' ?> - <?= strtoupper($outlet) ?></title>
        <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            .size-price-group {
                border: 1px solid #ddd;
                padding: 15px;
                border-radius: 8px;
                margin-bottom: 20px;
                background: #f9f9f9;
            }
            .size-option {
                margin-bottom: 15px;
            }
            .size-option label {
                font-weight: 600;
                margin-bottom: 5px;
                display: block;
            }
            .image-preview {
                max-width: 200px;
                max-height: 150px;
                margin: 10px 0;
                border: 1px solid #ddd;
                border-radius: 5px;
                padding: 5px;
            }
            .form-container {
                max-width: 1000px;
                margin: 0 auto;
                padding: 20px;
            }
            .card {
                border: none;
                box-shadow: 0 0 20px rgba(0,0,0,0.1);
            }
            .pizza-only {
                display: <?= $isPizza ? 'block' : 'none' ?>;
            }
        </style>
    </head>
    <body class="bg-light">
    <div class="form-container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h3><i class="fas <?= $isPizza ? 'fa-pizza-slice' : 'fa-utensils' ?>"></i> 
                <?= $id ? 'Edit' : 'Add New' ?> <?= $isPizza ? 'Pizza' : 'Menu Item' ?></h3>
            <a href="menu.php?outlet=<?= $outlet ?>" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to Menu
            </a>
        </div>
        
        <?php if (isset($_SESSION['error_message'])): ?>
            <div class="alert alert-danger">
                <?= $_SESSION['error_message'] ?>
                <?php unset($_SESSION['error_message']); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="alert alert-success">
                <?= $_SESSION['success_message'] ?>
                <?php unset($_SESSION['success_message']); ?>
            </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-body">
                <form method="POST" action="menu_save.php" enctype="multipart/form-data">
                    <input type="hidden" name="id" value="<?= htmlspecialchars($item['id'] ?? '') ?>">
                    <input type="hidden" name="outlet" value="<?= htmlspecialchars($outlet) ?>">
                    <input type="hidden" name="action" value="<?= $action ?>">
                    <input type="hidden" name="item_type" value="<?= $isPizza ? 'pizza' : 'regular' ?>">
                    <input type="hidden" name="current_image" value="<?= htmlspecialchars($item['image_url'] ?? '') ?>">
                    
                    <div class="row">
                        <div class="col-md-8">
                            <!-- Basic Info -->
                            <div class="mb-3">
                                <label class="form-label"><?= $isPizza ? 'Pizza' : 'Item' ?> Name *</label>
                                <input type="text" class="form-control" name="name" 
                                       value="<?= htmlspecialchars($item['name']) ?>" required
                                       placeholder="<?= $isPizza ? 'e.g., Margherita Pizza' : 'e.g., Caesar Salad' ?>">
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Description</label>
                                <textarea class="form-control" name="description" rows="3" 
                                          placeholder="Describe the <?= $isPizza ? 'pizza ingredients' : 'item' ?> and features"><?= 
                                    htmlspecialchars($item['description']) ?></textarea>
                            </div>
                            
                            <!-- Category Selection -->
                            <?php if ($isPizza): ?>
                                <!-- Pizza Category -->
                                <div class="mb-3 pizza-only">
                                    <label class="form-label">Pizza Category *</label>
                                    <select class="form-control" name="pizza_category_id" required>
                                        <option value="">Select Pizza Category</option>
                                        <?php foreach ($pizzaCategories as $cat): ?>
                                        <option value="<?= $cat['id'] ?>" 
                                                <?= (isset($item['pizza_category_id']) && $item['pizza_category_id'] == $cat['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($cat['name']) ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">Classic, Premium, or Special pizzas</small>
                                    <input type="hidden" name="category" value="pizza">
                                </div>
                            <?php else: ?>
                                <!-- Regular Item Category -->
                                <div class="mb-3">
                                    <label class="form-label">Category *</label>
                                    <select class="form-control" name="category" required>
                                        <?php foreach ($categories as $cat): ?>
                                        <option value="<?= htmlspecialchars($cat) ?>" 
                                                <?= ($item['category'] == $cat) ? 'selected' : '' ?>>
                                            <?= ucfirst(htmlspecialchars($cat)) ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <div id="otherCategoryInput" class="mt-2" style="display: <?= $item['category'] === 'other' ? 'block' : 'none' ?>;">
                                        <input type="text" class="form-control" name="other_category" 
                                               value="<?= $item['category'] === 'other' ? htmlspecialchars($item['category']) : '' ?>"
                                               placeholder="Enter new category name">
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Size Pricing (Pizza Only) -->
                            <?php if ($isPizza): ?>
                            <div class="size-price-group pizza-only">
                                <h5><i class="fas fa-ruler-combined"></i> Pizza Size Pricing</h5>
                                <p class="text-muted">Set prices for each pizza size. Leave 0 to disable a size.</p>
                                
                                <div class="row">
                                    <div class="col-md-3 size-option">
                                        <label>Small Size Price (KES)</label>
                                        <input type="number" class="form-control" 
                                               name="sizes[small]" 
                                               value="<?= $defaultSizes['small'] ?>" 
                                               min="0" step="50" 
                                               placeholder="e.g., 450">
                                        <small class="text-muted">Regular size</small>
                                    </div>
                                    <div class="col-md-3 size-option">
                                        <label>Medium Size Price (KES)</label>
                                        <input type="number" class="form-control" 
                                               name="sizes[medium]" 
                                               value="<?= $defaultSizes['medium'] ?>" 
                                               min="0" step="50" 
                                               placeholder="e.g., 650">
                                        <small class="text-muted">40-50% more than small</small>
                                    </div>
                                    <div class="col-md-3 size-option">
                                        <label>Large Size Price (KES)</label>
                                        <input type="number" class="form-control" 
                                               name="sizes[large]" 
                                               value="<?= $defaultSizes['large'] ?>" 
                                               min="0" step="50" 
                                               placeholder="e.g., 850">
                                        <small class="text-muted">80-100% more than small</small>
                                    </div>
                                    <div class="col-md-3 size-option">
                                        <label>Giant Size Price (KES)</label>
                                        <input type="number" class="form-control" 
                                               name="sizes[giant]" 
                                               value="<?= $defaultSizes['giant'] ?>" 
                                               min="0" step="50" 
                                               placeholder="e.g., 1050">
                                        <small class="text-muted">120-150% more than small</small>
                                    </div>
                                </div>
                                
                                <div class="mt-3">
                                    <button type="button" class="btn btn-sm btn-outline-secondary" id="autoFillBtn">
                                        <i class="fas fa-calculator"></i> Auto-fill based on Small size
                                    </button>
                                    <small class="text-muted ms-2">Click after entering Small price</small>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Price -->
                            <div class="mb-3">
                                <label class="form-label">Price (KES) *</label>
                                <input type="number" class="form-control" name="price" 
                                       value="<?= htmlspecialchars($item['price']) ?>" 
                                       min="0" step="50" required
                                       placeholder="<?= $isPizza ? 'Price for Small size' : 'Item price' ?>">
                                <?php if ($isPizza): ?>
                                <small class="text-muted">This is used as the Small size price if not set above</small>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <!-- Image Upload -->
                            <div class="mb-3">
                                <label class="form-label"><?= $isPizza ? 'Pizza' : 'Item' ?> Image</label>
                                <?php if (!empty($item['image_url'])): ?>
                                <div class="image-preview mb-2">
                                    <img src="../images/<?= htmlspecialchars($item['image_url']) ?>" 
                                         alt="Current image" class="img-fluid">
                                    <div class="mt-1 text-center">
                                        <small>Current image</small>
                                    </div>
                                </div>
                                <?php endif; ?>
                                <input type="file" class="form-control" name="image" accept="image/*">
                                <small class="text-muted">Recommended: 600x400 pixels, JPG/PNG format</small>
                            </div>
                            
                            <!-- Status -->
                            <div class="mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" 
                                           name="is_active" id="isActive" 
                                           <?= (($item['is_active'] ?? 1) == 1) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="isActive">
                                        Active (available for ordering)
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Submit -->
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="fas fa-save"></i> Save <?= $isPizza ? 'Pizza' : 'Item' ?>
                                </button>
                                <a href="menu.php?outlet=<?= $outlet ?>" class="btn btn-outline-secondary">
                                    Cancel
                                </a>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
    // Show/hide other category input
    document.querySelector('select[name="category"]')?.addEventListener('change', function() {
        const otherInput = document.getElementById('otherCategoryInput');
        if (this.value === 'other') {
            otherInput.style.display = 'block';
            if (otherInput.querySelector('input[name="other_category"]')) {
                otherInput.querySelector('input[name="other_category"]').focus();
            }
        } else {
            otherInput.style.display = 'none';
        }
    });
    
    <?php if ($isPizza): ?>
    // Pizza-specific JavaScript
    // Auto-fill size prices based on small size
    document.getElementById('autoFillBtn')?.addEventListener('click', function() {
        const smallPrice = document.querySelector('input[name="sizes[small]"]').value;
        const basePrice = document.querySelector('input[name="price"]').value;
        
        // Use small price if set, otherwise use base price
        const price = parseFloat(smallPrice) || parseFloat(basePrice) || 0;
        
        if (price > 0) {
            // Calculate other sizes (adjust multipliers as needed)
            const mediumPrice = Math.round(price * 1.4 / 50) * 50; // 40% more
            const largePrice = Math.round(price * 1.8 / 50) * 50;  // 80% more
            const giantPrice = Math.round(price * 2.2 / 50) * 50;  // 120% more
            
            // Fill the fields
            document.querySelector('input[name="sizes[medium]"]').value = mediumPrice;
            document.querySelector('input[name="sizes[large]"]').value = largePrice;
            document.querySelector('input[name="sizes[giant]"]').value = giantPrice;
            
            alert('Size prices auto-filled based on small price!');
        } else {
            alert('Please enter a Small size or Base price first.');
        }
    });
    
    // Auto-update small price when base price changes (if small is empty)
    document.querySelector('input[name="price"]')?.addEventListener('change', function() {
        const smallInput = document.querySelector('input[name="sizes[small]"]');
        if (!smallInput.value || smallInput.value == 0) {
            smallInput.value = this.value;
        }
    });
    
    // Auto-update base price when small price changes
    document.querySelector('input[name="sizes[small]"]')?.addEventListener('change', function() {
        const baseInput = document.querySelector('input[name="price"]');
        if (this.value > 0) {
            baseInput.value = this.value;
        }
    });
    <?php endif; ?>
    </script>
    </body>
    </html>
    <?php
    exit;
}

// --------------------------------------------------
// LIST VIEW (Default)
// --------------------------------------------------

// Load items for the current outlet
try {
    $stmt = $conn->prepare("SELECT * FROM menu_items WHERE outlet=? ORDER BY category, sort_order, name");
    $stmt->execute([$outlet]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $items = [];
    $_SESSION['error_message'] = "Error loading menu: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Menu Admin - <?= strtoupper($outlet) ?></title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background: #f8f9fa;
        }
        .header-container {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .menu-card {
            border: none;
            border-radius: 10px;
            margin-bottom: 15px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            transition: transform 0.2s;
        }
        .menu-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.15);
        }
        .category-badge {
            background: #D35400;
            color: white;
            padding: 3px 10px;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .pizza-badge {
            background: #27AE60;
            color: white;
            padding: 3px 10px;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .status-badge {
            padding: 3px 10px;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        .status-active {
            background: #d4edda;
            color: #155724;
        }
        .status-inactive {
            background: #f8d7da;
            color: #721c24;
        }
        .price-tag {
            font-weight: 700;
            color: #D35400;
            font-size: 1.1rem;
        }
        .action-buttons {
            display: flex;
            gap: 10px;
        }
        .image-thumb {
            width: 60px;
            height: 60px;
            object-fit: cover;
            border-radius: 5px;
            border: 1px solid #dee2e6;
        }
        .empty-state {
            text-align: center;
            padding: 50px 20px;
            color: #6c757d;
        }
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 15px;
            opacity: 0.3;
        }
        .type-filter {
            background: white;
            padding: 10px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .menu-item {
            display: block;
        }
    </style>
</head>
<body>

<div class="container-fluid mt-4">
    <div class="header-container">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h3><i class="fas fa-<?= $outlet === 'milano' ? 'pizza-slice' : 'fish' ?>"></i> 
                    <?= strtoupper($outlet) ?> MENU ADMIN</h3>
                <p class="text-muted mb-0">Manage your menu items and pizza sizes</p>
            </div>
            <div class="action-buttons">
                <a href="?outlet=<?= $outlet ?>&action=add&type=pizza" class="btn btn-primary">
                    <i class="fas fa-pizza-slice"></i> Add Pizza
                </a>
                <a href="?outlet=<?= $outlet ?>&action=add&type=regular" class="btn btn-outline-primary">
                    <i class="fas fa-plus"></i> Add Other Item
                </a>
            </div>
        </div>
        
        <!-- Outlet Selector -->
        <div class="mt-3">
            <div class="btn-group" role="group">
                <a href="?outlet=milano" class="btn <?= $outlet === 'milano' ? 'btn-warning' : 'btn-outline-warning' ?>">
                    <i class="fas fa-pizza-slice"></i> Milano Pizza
                </a>
                <a href="?outlet=sushi" class="btn <?= $outlet === 'sushi' ? 'btn-primary' : 'btn-outline-primary' ?>">
                    <i class="fas fa-fish"></i> Sushi Bar
                </a>
            </div>
        </div>
    </div>
    
    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle"></i> <?= $_SESSION['error_message'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle"></i> <?= $_SESSION['success_message'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>
    
    <!-- Stats -->
    <?php
    $pizzaCount = count(array_filter($items, fn($i) => $i['category'] === 'pizza'));
    $regularCount = count($items) - $pizzaCount;
    $activeCount = count(array_filter($items, fn($i) => $i['is_active'] == 1));
    $sizeCount = count(array_filter($items, fn($i) => !empty($i['sizes'])));
    ?>
    
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card text-white bg-primary">
                <div class="card-body">
                    <h5 class="card-title"><?= count($items) ?></h5>
                    <p class="card-text">Total Items</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-white bg-success">
                <div class="card-body">
                    <h5 class="card-title"><?= $pizzaCount ?></h5>
                    <p class="card-text">Pizzas</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-white bg-info">
                <div class="card-body">
                    <h5 class="card-title"><?= $regularCount ?></h5>
                    <p class="card-text">Other Items</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-white bg-warning">
                <div class="card-body">
                    <h5 class="card-title"><?= $activeCount ?></h5>
                    <p class="card-text">Active Items</p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Type Filter -->
    <div class="type-filter mb-3">
        <div class="btn-group" role="group">
            <button type="button" class="btn btn-outline-secondary active" onclick="filterItems('all')">
                <i class="fas fa-list"></i> All Items (<?= count($items) ?>)
            </button>
            <button type="button" class="btn btn-outline-secondary" onclick="filterItems('pizza')">
                <i class="fas fa-pizza-slice"></i> Pizzas (<?= $pizzaCount ?>)
            </button>
            <button type="button" class="btn btn-outline-secondary" onclick="filterItems('regular')">
                <i class="fas fa-utensils"></i> Other Items (<?= $regularCount ?>)
            </button>
        </div>
    </div>
    
    <!-- Menu Items List -->
    <?php if (empty($items)): ?>
        <div class="empty-state">
            <i class="fas fa-utensils"></i>
            <h4>No menu items yet</h4>
            <p>Start by adding your first item!</p>
            <div class="mt-3">
                <a href="?outlet=<?= $outlet ?>&action=add&type=pizza" class="btn btn-primary btn-lg me-2">
                    <i class="fas fa-pizza-slice"></i> Add First Pizza
                </a>
                <a href="?outlet=<?= $outlet ?>&action=add&type=regular" class="btn btn-outline-primary btn-lg">
                    <i class="fas fa-plus"></i> Add Other Item
                </a>
            </div>
        </div>
    <?php else: ?>
        <div class="row" id="menuItemsContainer">
            <?php foreach ($items as $item): 
                $isPizza = ($item['category'] === 'pizza');
                $itemTypeClass = $isPizza ? 'pizza-item' : 'regular-item';
            ?>
            <div class="col-md-6 menu-item <?= $itemTypeClass ?>">
                <div class="menu-card">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-2">
                                <?php if (!empty($item['image_url'])): ?>
                                    <img src="../images/<?= htmlspecialchars($item['image_url']) ?>" 
                                         class="image-thumb" 
                                         alt="<?= htmlspecialchars($item['name']) ?>">
                                <?php else: ?>
                                    <div class="image-thumb bg-light d-flex align-items-center justify-content-center">
                                        <i class="fas fa-<?= $isPizza ? 'pizza-slice' : 'utensils' ?> text-muted"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="col-6">
                                <h6 class="mb-1"><?= htmlspecialchars($item['name']) ?></h6>
                                <div class="mb-1">
                                    <span class="category-badge"><?= htmlspecialchars($item['category']) ?></span>
                                    <?php if ($isPizza): ?>
                                        <span class="pizza-badge ms-1">Pizza</span>
                                    <?php endif; ?>
                                    <span class="status-badge status-<?= $item['is_active'] ? 'active' : 'inactive' ?> ms-1">
                                        <?= $item['is_active'] ? 'Active' : 'Inactive' ?>
                                    </span>
                                </div>
                                <p class="text-muted mb-0 small" style="font-size: 0.85rem;">
                                    <?= substr(htmlspecialchars($item['description'] ?? ''), 0, 80) ?><?= 
                                        strlen($item['description'] ?? '') > 80 ? '...' : '' ?>
                                </p>
                            </div>
                            <div class="col-2 text-center">
                                <div class="price-tag">KES <?= number_format($item['price'], 0) ?></div>
                                <?php if (!empty($item['sizes'])): ?>
                                    <small class="text-success">Has sizes</small>
                                <?php endif; ?>
                            </div>
                            <div class="col-2">
                                <div class="action-buttons">
                                    <a href="?outlet=<?= $outlet ?>&action=edit&id=<?= $item['id'] ?>&type=<?= $isPizza ? 'pizza' : 'regular' ?>" 
                                       class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <button class="btn btn-sm btn-outline-danger" 
                                            onclick="confirmDelete(<?= $item['id'] ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Filter items by type
function filterItems(type) {
    const buttons = document.querySelectorAll('.type-filter .btn');
    buttons.forEach(btn => {
        btn.classList.remove('active');
        if (btn.textContent.includes(type === 'all' ? 'All Items' : 
                                    type === 'pizza' ? 'Pizzas' : 'Other Items')) {
            btn.classList.add('active');
        }
    });
    
    const items = document.querySelectorAll('.menu-item');
    items.forEach(item => {
        if (type === 'all') {
            item.style.display = 'block';
        } else if (type === 'pizza') {
            item.style.display = item.classList.contains('pizza-item') ? 'block' : 'none';
        } else if (type === 'regular') {
            item.style.display = item.classList.contains('regular-item') ? 'block' : 'none';
        }
    });
}

function confirmDelete(itemId) {
    if (confirm('Are you sure you want to delete this item? This action cannot be undone.')) {
        // Submit form or make AJAX call
        fetch('delete_item.php?id=' + itemId)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                alert('Network error: ' + error);
            });
    }
}

// Initialize Bootstrap tooltips
var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
    return new bootstrap.Tooltip(tooltipTriggerEl);
});
</script>
</body>
</html>