<?php
// admin/profile.php - Admin Profile

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$admin_info = $auth->getAdminInfo();
$database = new Database();
$conn = $database->getConnection();

// Get admin details
$stmt = $conn->prepare("SELECT * FROM admin_users WHERE id = :id");
$stmt->execute([':id' => $admin_info['id']]);
$admin = $stmt->fetch(PDO::FETCH_ASSOC);

$error = '';
$success = '';

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    $name = trim($_POST['name']);
    $email = trim($_POST['email']);
    
    if (empty($name)) {
        $error = 'Name is required';
    } else {
        $stmt = $conn->prepare("UPDATE admin_users SET name = :name, email = :email, updated_at = NOW() WHERE id = :id");
        $stmt->execute([
            ':name' => $name,
            ':email' => $email,
            ':id' => $admin_info['id']
        ]);
        
        // Update session
        $_SESSION['admin_name'] = $name;
        
        $auth->logActivity($admin_info['id'], 'profile_update', 'Updated profile information');
        $success = 'Profile updated successfully!';
        
        // Refresh admin data
        $stmt = $conn->prepare("SELECT * FROM admin_users WHERE id = :id");
        $stmt->execute([':id' => $admin_info['id']]);
        $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $error = 'All password fields are required';
    } elseif ($new_password !== $confirm_password) {
        $error = 'New passwords do not match';
    } elseif (strlen($new_password) < 6) {
        $error = 'Password must be at least 6 characters';
    } else {
        if ($auth->changePassword($admin_info['id'], $current_password, $new_password)) {
            $success = 'Password changed successfully!';
        } else {
            $error = 'Current password is incorrect';
        }
    }
}

// Get activity logs
$stmt = $conn->prepare("SELECT * FROM admin_activity_log WHERE admin_id = :id ORDER BY created_at DESC LIMIT 10");
$stmt->execute([':id' => $admin_info['id']]);
$activity_logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<?php include 'header.php'; ?>

<div class="container-fluid mt-4">
    <div class="row">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Profile Information</h5>
                </div>
                <div class="card-body">
                    <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" value="<?php echo htmlspecialchars($admin['username']); ?>" readonly>
                                <small class="text-muted">Username cannot be changed</small>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Role</label>
                                <input type="text" class="form-control" value="<?php echo ucfirst($admin['role']); ?>" readonly>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Full Name *</label>
                                <input type="text" name="name" class="form-control" 
                                       value="<?php echo htmlspecialchars($admin['name']); ?>" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Email</label>
                                <input type="email" name="email" class="form-control" 
                                       value="<?php echo htmlspecialchars($admin['email'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Last Login</label>
                                <input type="text" class="form-control" 
                                       value="<?php echo $admin['last_login'] ? date('F j, Y g:i A', strtotime($admin['last_login'])) : 'Never'; ?>" readonly>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Account Created</label>
                                <input type="text" class="form-control" 
                                       value="<?php echo date('F j, Y', strtotime($admin['created_at'])); ?>" readonly>
                            </div>
                            
                            <div class="col-12 mt-4">
                                <button type="submit" name="update_profile" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Update Profile
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Password Change Form -->
            <div class="card mt-4">
                <div class="card-header">
                    <h5 class="mb-0">Change Password</h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Current Password *</label>
                                <input type="password" name="current_password" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">New Password *</label>
                                <input type="password" name="new_password" class="form-control" required>
                                <small class="text-muted">Minimum 6 characters</small>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Confirm New Password *</label>
                                <input type="password" name="confirm_password" class="form-control" required>
                            </div>
                            
                            <div class="col-12 mt-4">
                                <button type="submit" name="change_password" class="btn btn-primary">
                                    <i class="fas fa-key"></i> Change Password
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Sidebar -->
        <div class="col-md-4">
            <!-- Activity Logs -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Recent Activity</h5>
                </div>
                <div class="card-body">
                    <div class="timeline" style="max-height: 400px; overflow-y: auto;">
                        <?php if (empty($activity_logs)): ?>
                        <p class="text-muted text-center">No recent activity</p>
                        <?php else: ?>
                        <?php foreach ($activity_logs as $log): ?>
                        <div class="timeline-item mb-3">
                            <strong><?php echo htmlspecialchars($log['action']); ?></strong>
                            <p class="mb-1"><?php echo htmlspecialchars($log['description']); ?></p>
                            <small class="text-muted">
                                <i class="fas fa-clock"></i> 
                                <?php echo date('M j, g:i A', strtotime($log['created_at'])); ?>
                            </small>
                        </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                    <div class="text-center mt-3">
                        <a href="activity_logs.php" class="btn btn-sm btn-outline-primary">
                            View All Activity
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Account Status -->
            <div class="card mt-4">
                <div class="card-header">
                    <h5 class="mb-0">Account Status</h5>
                </div>
                <div class="card-body">
                    <div class="d-flex align-items-center mb-3">
                        <div class="me-3">
                            <div class="rounded-circle bg-<?php echo $admin['status'] === 'active' ? 'success' : ($admin['status'] === 'suspended' ? 'danger' : 'warning'); ?>" 
                                 style="width: 20px; height: 20px;"></div>
                        </div>
                        <div>
                            <strong>Status:</strong> 
                            <span class="badge bg-<?php echo $admin['status'] === 'active' ? 'success' : ($admin['status'] === 'suspended' ? 'danger' : 'warning'); ?>">
                                <?php echo ucfirst($admin['status']); ?>
                            </span>
                        </div>
                    </div>
                    
                    <?php if ($admin['status'] !== 'active'): ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        Your account is not active. Contact administrator for assistance.
                    </div>
                    <?php endif; ?>
                    
                    <div class="d-grid gap-2">
                        <a href="logout.php" class="btn btn-outline-danger" 
                           onclick="return confirm('Are you sure you want to logout from all devices?');">
                            <i class="fas fa-sign-out-alt"></i> Logout All Devices
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>