<?php
// check_payment_status.php - Updated for your callback structure

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/config/database.php';

header('Content-Type: application/json');

// Get order ID and checkout ID from request
$order_id = $_GET['order_id'] ?? null;
$checkout_id = $_GET['checkout_id'] ?? null;

if (!$order_id && !$checkout_id) {
    echo json_encode([
        'payment_status' => 'error',
        'message' => 'Missing order ID or checkout ID'
    ]);
    exit;
}

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    // Build query based on what we have
    $sql = "SELECT 
                o.id,
                o.order_number,
                o.status as order_status,
                o.payment_method,
                o.mpesa_checkout_id,
                o.mpesa_receipt,
                o.total_amount,
                o.delivery_location,
                o.customer_name,
                o.customer_phone,
                mt.status as mpesa_status,
                mt.result_code,
                mt.result_desc,
                mt.mpesa_receipt_number,
                mt.transaction_date
            FROM orders o
            LEFT JOIN mpesa_transactions mt ON o.mpesa_checkout_id = mt.checkout_request_id 
            WHERE ";
    
    $params = [];
    
    if ($order_id) {
        $sql .= "o.id = :order_id";
        $params[':order_id'] = $order_id;
    } else {
        $sql .= "o.mpesa_checkout_id = :checkout_id";
        $params[':checkout_id'] = $checkout_id;
    }
    
    $sql .= " ORDER BY mt.id DESC LIMIT 1";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode([
            'payment_status' => 'not_found',
            'message' => 'Order not found'
        ]);
        exit;
    }
    
    // Determine payment status based on order status and M-Pesa transaction
    $payment_status = 'pending';
    $message = 'Payment is pending';
    
    if ($order['order_status'] === 'paid' || $order['mpesa_receipt']) {
        $payment_status = 'completed';
        $message = 'Payment completed successfully';
    } elseif ($order['order_status'] === 'cancelled' || $order['order_status'] === 'failed') {
        $payment_status = 'failed';
        $message = 'Payment failed or was cancelled';
    } elseif ($order['mpesa_status'] === 'success') {
        $payment_status = 'completed';
        $message = 'Payment completed via M-Pesa';
    } elseif ($order['mpesa_status'] === 'failed') {
        $payment_status = 'failed';
        $message = 'M-Pesa payment failed';
    }
    
    // For M-Pesa payments that are still pending, check if we have a receipt
    if ($payment_status === 'pending' && $order['payment_method'] === 'mpesa') {
        // Check if we have any M-Pesa transaction record
        $check_sql = "SELECT COUNT(*) as count FROM mpesa_transactions 
                      WHERE checkout_request_id = :checkout_id 
                      AND status = 'success'";
        $check_stmt = $conn->prepare($check_sql);
        $check_stmt->execute([':checkout_id' => $order['mpesa_checkout_id']]);
        $check_result = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($check_result['count'] > 0) {
            $payment_status = 'completed';
            $message = 'Payment completed (callback received)';
        }
    }
    
    echo json_encode([
        'payment_status' => $payment_status,
        'order_status' => $order['order_status'],
        'order_number' => $order['order_number'],
        'order_id' => $order['id'],
        'payment_method' => $order['payment_method'],
        'mpesa_receipt' => $order['mpesa_receipt'] ?? $order['mpesa_receipt_number'] ?? null,
        'transaction_date' => $order['transaction_date'] ?? null,
        'total_amount' => $order['total_amount'],
        'delivery_location' => $order['delivery_location'],
        'customer_name' => $order['customer_name'],
        'customer_phone' => $order['customer_phone'],
        'message' => $message,
        'result_desc' => $order['result_desc'] ?? null
    ]);
    
} catch (Exception $e) {
    error_log("Payment status check error: " . $e->getMessage());
    
    echo json_encode([
        'payment_status' => 'error',
        'message' => 'System error: ' . $e->getMessage()
    ]);
}
?>