<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SUSHI-BAR | Online Ordering</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* All your existing CSS styles remain the same */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        :root {
            --primary: #e53935;
            --primary-dark: #c62828;
            --secondary: #25D366;
            --whatsapp: #25D366;
            --light: #f5f5f5;
            --dark: #333;
            --gray: #777;
            --light-gray: #f0f0f0;
            --border: #ddd;
            --shadow: 0 4px 12px rgba(0,0,0,0.1);
            --shadow-lg: 0 8px 25px rgba(0,0,0,0.15);
        }
        
        body {
            background-color: #f9f9f9;
            color: var(--dark);
            line-height: 1.6;
            padding-bottom: 100px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        /* Header Styles */
        header {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            padding: 20px 0;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .logo i {
            font-size: 2.5rem;
        }
        
        .logo h1 {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .logo span {
            color: #ffeb3b;
        }
        
        .header-actions {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .whatsapp-order-btn {
            background: var(--whatsapp);
            color: white;
            padding: 10px 20px;
            border-radius: 30px;
            text-decoration: none;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .whatsapp-order-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(37, 211, 102, 0.3);
        }
        
        .cart-icon {
            position: relative;
            font-size: 1.5rem;
            cursor: pointer;
        }
        
        .cart-count {
            position: absolute;
            top: -8px;
            right: -8px;
            background: var(--secondary);
            color: white;
            border-radius: 50%;
            width: 22px;
            height: 22px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8rem;
            font-weight: bold;
        }
        
        /* Floating Cart */
        .floating-cart {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            padding: 15px 20px;
            box-shadow: 0 -4px 15px rgba(0,0,0,0.1);
            z-index: 1000;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-top: 3px solid var(--primary);
            transform: translateY(100%);
            transition: transform 0.3s ease;
        }
        
        .floating-cart.visible {
            transform: translateY(0);
        }
        
        .cart-summary {
            display: flex;
            gap: 30px;
            align-items: center;
        }
        
        .cart-summary-item {
            display: flex;
            flex-direction: column;
        }
        
        .cart-summary-label {
            font-size: 0.9rem;
            color: var(--gray);
        }
        
        .cart-summary-value {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        .cart-actions {
            display: flex;
            gap: 10px;
        }
        
        .floating-checkout-btn, .empty-cart-btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .floating-checkout-btn {
            background: var(--primary);
            color: white;
        }
        
        .empty-cart-btn {
            background: var(--light-gray);
            color: var(--dark);
        }
        
        .floating-checkout-btn:hover {
            background: var(--primary-dark);
        }
        
        .empty-cart-btn:hover {
            background: #e0e0e0;
        }
        
        /* Main Layout */
        .main-content {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
            margin: 30px 0;
        }
        
        /* Menu Section */
        .menu-section {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: var(--shadow);
        }
        
        .section-title {
            font-size: 1.8rem;
            margin-bottom: 25px;
            padding-bottom: 10px;
            border-bottom: 3px solid var(--primary);
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .category-filters {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
            flex-wrap: wrap;
        }
        
        .category-btn {
            padding: 8px 20px;
            background: var(--light-gray);
            border: none;
            border-radius: 30px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .category-btn.active,
        .category-btn:hover {
            background: var(--primary);
            color: white;
        }
        
        .menu-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 25px;
        }
        
        .menu-item {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
            transition: transform 0.3s, box-shadow 0.3s;
            border: 1px solid var(--border);
        }
        
        .menu-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.12);
        }
        
        .item-image {
            height: 180px;
            background: var(--light-gray);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--gray);
            position: relative;
            overflow: hidden;
        }
        
        .item-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .item-image i {
            font-size: 3rem;
            opacity: 0.3;
        }
        
        .item-category {
            position: absolute;
            top: 15px;
            right: 15px;
            background: rgba(0,0,0,0.7);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
            z-index: 2;
        }
        
        .item-details {
            padding: 20px;
        }
        
        .item-name {
            font-size: 1.3rem;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .item-description {
            color: var(--gray);
            font-size: 0.95rem;
            margin-bottom: 15px;
            line-height: 1.5;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .item-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .item-price {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        .add-to-cart {
            background: var(--secondary);
            color: white;
            border: none;
            border-radius: 5px;
            padding: 8px 15px;
            cursor: pointer;
            font-weight: 600;
            transition: background 0.3s;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .add-to-cart:hover {
            background: #1b8b3c;
        }
        
        .add-to-cart:disabled {
            background: #cccccc;
            cursor: not-allowed;
        }
        
        /* Order Summary */
        .order-summary {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 100px;
            height: fit-content;
        }
        
        .order-items {
            max-height: 300px;
            overflow-y: auto;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--border);
            padding-bottom: 20px;
        }
        
        .order-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px dashed var(--border);
        }
        
        .order-item:last-child {
            border-bottom: none;
        }
        
        .order-item-name {
            font-weight: 600;
            flex: 1;
        }
        
        .order-item-controls {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .quantity-btn {
            width: 28px;
            height: 28px;
            border-radius: 50%;
            border: none;
            background: var(--light-gray);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        
        .quantity-btn:hover {
            background: var(--primary);
            color: white;
        }
        
        .quantity-btn:disabled {
            background: #f0f0f0;
            color: #999;
            cursor: not-allowed;
        }
        
        .item-quantity {
            font-weight: 600;
            min-width: 30px;
            text-align: center;
        }
        
        .order-item-price {
            font-weight: 600;
            color: var(--primary);
            min-width: 70px;
            text-align: right;
        }
        
        .remove-item {
            color: var(--primary);
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.2rem;
            margin-left: 10px;
        }
        
        .remove-item:hover {
            color: var(--primary-dark);
        }
        
        .order-totals {
            margin-bottom: 25px;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px dashed var(--border);
        }
        
        .total-row:last-child {
            border-bottom: none;
            font-size: 1.3rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        .cart-actions-main {
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .checkout-btn, .empty-cart-main-btn {
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.3s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .checkout-btn {
            background: var(--primary);
            color: white;
        }
        
        .checkout-btn:hover {
            background: var(--primary-dark);
        }
        
        .checkout-btn:disabled {
            background: #cccccc;
            cursor: not-allowed;
        }
        
        .empty-cart-main-btn {
            background: var(--light-gray);
            color: var(--dark);
        }
        
        .empty-cart-main-btn:hover {
            background: #e0e0e0;
        }
        
        .empty-cart {
            text-align: center;
            padding: 40px 20px;
            color: var(--gray);
        }
        
        .empty-cart i {
            font-size: 3rem;
            margin-bottom: 15px;
            opacity: 0.3;
        }
        
        /* Checkout Modal */
        .checkout-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 2000;
            padding: 20px;
        }
        
        .checkout-modal.active {
            display: flex;
        }
        
        .checkout-content {
            background: white;
            border-radius: 10px;
            width: 100%;
            max-width: 500px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-lg);
            animation: fadeIn 0.3s ease-out;
        }
        
        .checkout-header {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            padding: 20px;
            border-radius: 10px 10px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .checkout-header h2 {
            font-size: 1.5rem;
        }
        
        .close-checkout {
            background: none;
            border: none;
            color: white;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 5px;
        }
        
        .checkout-body {
            padding: 25px;
        }
        
        .checkout-step {
            display: none;
        }
        
        .checkout-step.active {
            display: block;
        }
        
        .step-title {
            font-size: 1.3rem;
            margin-bottom: 20px;
            color: var(--dark);
            padding-bottom: 10px;
            border-bottom: 2px solid var(--primary);
        }
        
        .delivery-options {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
        }
        
        .delivery-option {
            border: 2px solid var(--border);
            border-radius: 8px;
            padding: 15px;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .delivery-option:hover {
            border-color: var(--primary);
            background-color: rgba(229, 57, 53, 0.05);
        }
        
        .delivery-option.selected {
            border-color: var(--secondary);
            background-color: rgba(46, 125, 50, 0.1);
        }
        
        .location-name {
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        .delivery-cost {
            font-weight: 700;
            color: var(--primary);
        }
        
        .delivery-cost.free {
            color: var(--secondary);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid var(--border);
            border-radius: 5px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        .form-control:invalid {
            border-color: #ff4444;
        }
        
        .checkout-footer {
            display: flex;
            justify-content: space-between;
            gap: 10px;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid var(--border);
        }
        
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 5px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-primary:hover {
            background: var(--primary-dark);
        }
        
        .btn-secondary {
            background: var(--light-gray);
            color: var(--dark);
        }
        
        .btn-secondary:hover {
            background: #e0e0e0;
        }
        
        /* Loading Overlay */
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 3000;
            color: white;
            display: none;
        }
        
        .spinner {
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--primary);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
            margin-bottom: 20px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Error Message */
        .error-message {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #ff4444;
            color: white;
            padding: 15px 25px;
            border-radius: 5px;
            box-shadow: var(--shadow);
            z-index: 1000;
            animation: fadeIn 0.3s ease-out;
            display: flex;
            align-items: center;
            gap: 10px;
            max-width: 400px;
        }
        
        /* Success Message */
        .success-message {
            position: fixed;
            top: 20px;
            right: 20px;
            background: var(--secondary);
            color: white;
            padding: 15px 25px;
            border-radius: 5px;
            box-shadow: var(--shadow);
            z-index: 1000;
            animation: fadeIn 0.3s ease-out;
            display: flex;
            align-items: center;
            gap: 10px;
            max-width: 400px;
        }
        
        /* Menu Loading State */
        .menu-loading {
            grid-column: 1 / -1;
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }
        
        .menu-loading .spinner {
            width: 40px;
            height: 40px;
            margin: 0 auto 20px;
        }
        
        /* Responsive - UPDATED FOR MOBILE FLOATING CART */
        @media (max-width: 992px) {
            .main-content {
                grid-template-columns: 1fr;
            }
            
            .order-summary {
                position: static;
            }
            
            .delivery-options {
                grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
            }
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .header-actions {
                width: 100%;
                justify-content: center;
            }
            
            .menu-grid {
                grid-template-columns: 1fr;
            }
            
            .cart-summary {
                flex-direction: column;
                gap: 10px;
                align-items: flex-start;
            }
            
            /* Mobile-specific floating cart styles - MAX ONE THIRD OF SCREEN HEIGHT */
            .floating-cart {
                padding: 10px 15px;
                max-height: 33vh; /* Maximum one third of screen height */
                overflow-y: auto; /* Enable scrolling if content overflows */
                flex-direction: column;
                gap: 8px;
                align-items: stretch;
            }
            
            .floating-cart .cart-summary {
                display: grid;
                grid-template-columns: repeat(2, 1fr);
                gap: 8px;
                width: 100%;
                padding: 5px 0;
            }
            
            .floating-cart .cart-summary-item {
                text-align: center;
                padding: 6px;
                background: rgba(255, 255, 255, 0.1);
                border-radius: 5px;
                min-width: 0; /* Allow items to shrink */
            }
            
            .floating-cart .cart-summary-label {
                font-size: 0.75rem;
                color: var(--gray);
                margin-bottom: 2px;
            }
            
            .floating-cart .cart-summary-value {
                font-size: 0.95rem;
                font-weight: 700;
                color: var(--primary);
            }
            
            .floating-cart .cart-actions {
                width: 100%;
                display: flex;
                gap: 6px;
                margin-top: 3px;
            }
            
            .floating-cart .empty-cart-btn,
            .floating-cart .floating-checkout-btn {
                padding: 10px 12px;
                font-size: 0.85rem;
                min-height: 44px; /* Minimum touch target size */
                flex: 1;
            }
            
            /* Reduce padding for better mobile experience */
            .container {
                padding: 0 15px;
            }
            
            /* Adjust main content spacing */
            .main-content {
                margin: 15px 0;
                gap: 15px;
            }
            
            /* Reduce menu item height on mobile */
            .menu-item {
                min-height: auto;
            }
            
            .item-image {
                height: 140px;
            }
            
            /* Adjust order summary for mobile */
            .order-summary {
                margin-bottom: 70px; /* Make room for floating cart */
            }
        }
        
        /* For very small mobile devices */
        @media (max-width: 480px) {
            .floating-cart {
                max-height: 30vh; /* Slightly smaller on very small screens */
                padding: 8px 10px;
            }
            
            .floating-cart .cart-summary {
                grid-template-columns: repeat(2, 1fr);
                gap: 6px;
            }
            
            .floating-cart .cart-summary-label {
                font-size: 0.7rem;
            }
            
            .floating-cart .cart-summary-value {
                font-size: 0.85rem;
            }
            
            .floating-cart .cart-actions {
                flex-direction: row;
                gap: 4px;
            }
            
            .floating-cart .empty-cart-btn,
            .floating-cart .floating-checkout-btn {
                padding: 8px 10px;
                font-size: 0.8rem;
                gap: 4px;
            }
            
            /* Ensure cart content is scrollable if it gets too tall */
            .floating-cart .cart-summary {
                max-height: 18vh;
                overflow-y: auto;
                padding-right: 3px;
            }
            
            /* Style scrollbar for floating cart */
            .floating-cart::-webkit-scrollbar {
                width: 3px;
            }
            
            .floating-cart::-webkit-scrollbar-track {
                background: rgba(0, 0, 0, 0.05);
                border-radius: 10px;
            }
            
            .floating-cart::-webkit-scrollbar-thumb {
                background: var(--primary);
                border-radius: 10px;
            }
            
            .order-summary {
                margin-bottom: 60px; /* Less space needed on very small screens */
            }
        }
        
        /* Tablet styles (unchanged from your current) */
        @media (min-width: 769px) and (max-width: 992px) {
            .floating-cart {
                /* Keep original tablet styling */
                max-height: none; /* Remove mobile height restriction */
                padding: 15px 20px;
                flex-direction: row; /* Original layout */
                align-items: center; /* Original layout */
            }
            
            .floating-cart .cart-summary {
                display: flex; /* Original layout */
                gap: 30px; /* Original layout */
                align-items: center; /* Original layout */
            }
            
            .floating-cart .cart-summary-item {
                text-align: left; /* Original layout */
                padding: 0; /* Original layout */
                background: transparent; /* Original layout */
            }
            
            .floating-cart .cart-summary-label {
                font-size: 0.9rem; /* Original size */
            }
            
            .floating-cart .cart-summary-value {
                font-size: 1.2rem; /* Original size */
            }
            
            .floating-cart .cart-actions {
                width: auto; /* Original layout */
                margin-top: 0; /* Original layout */
            }
            
            .floating-cart .empty-cart-btn,
            .floating-cart .floating-checkout-btn {
                padding: 12px 25px; /* Original padding */
                font-size: 1rem; /* Original size */
            }
        }
        
        /* Desktop styles (unchanged from your current) */
        @media (min-width: 993px) {
            .floating-cart {
                /* Keep original desktop styling */
                max-height: none; /* Remove mobile height restriction */
                padding: 15px 20px;
                flex-direction: row; /* Original layout */
                align-items: center; /* Original layout */
            }
            
            .floating-cart .cart-summary {
                display: flex; /* Original layout */
                gap: 30px; /* Original layout */
                align-items: center; /* Original layout */
            }
            
            .floating-cart .cart-summary-item {
                text-align: left; /* Original layout */
                padding: 0; /* Original layout */
                background: transparent; /* Original layout */
            }
            
            .floating-cart .cart-summary-label {
                font-size: 0.9rem; /* Original size */
            }
            
            .floating-cart .cart-summary-value {
                font-size: 1.2rem; /* Original size */
            }
            
            .floating-cart .cart-actions {
                width: auto; /* Original layout */
                margin-top: 0; /* Original layout */
            }
            
            .floating-cart .empty-cart-btn,
            .floating-cart .floating-checkout-btn {
                padding: 12px 25px; /* Original padding */
                font-size: 1rem; /* Original size */
            }
        }
        
        /* Animations */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .fade-in {
            animation: fadeIn 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from { transform: translateY(100%); }
            to { transform: translateY(0); }
        }
        
        .slide-up {
            animation: slideUp 0.3s ease-out;
        }
    </style>
</head>
<body>
    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="spinner"></div>
        <p id="loadingMessage">Processing your order...</p>
    </div>

    <!-- Header -->
<header>
    <div class="container header-content">
        <div class="logo">
            <img src="https://aromacafe.co.ke/blaqpremium/images/sushilogo.png" 
                 alt="SUSHI-BAR Logo" 
                 style="height: 70px; width: auto;">
            
        </div>
        <div class="header-actions">
            <a href="https://wa.me/254790644644?text=Hello%20SUSHI-BAR%20I%20would%20like%20to%20make%20an%20order" 
               target="_blank" 
               class="whatsapp-order-btn">
                <i class="fab fa-whatsapp"></i> ORDER ON WHATSAPP
            </a>
            <div class="cart-icon" id="cartIcon">
                <i class="fas fa-shopping-cart"></i>
                <div class="cart-count" id="cartCount">0</div>
            </div>
        </div>
    </div>
</header>
    <!-- Checkout Modal -->
    <div class="checkout-modal" id="checkoutModal">
        <div class="checkout-content">
            <div class="checkout-header">
                <h2>Complete Your Order</h2>
                <button class="close-checkout" id="closeCheckout">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="checkout-body">
                <!-- Step 1: Delivery Info -->
                <div class="checkout-step active" id="step1">
                    <h3 class="step-title">Delivery Information</h3>
                    
                    <div class="form-group">
                        <label for="customerName">Full Name *</label>
                        <input type="text" id="customerName" class="form-control" placeholder="Enter your full name" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="customerPhone">Phone Number *</label>
                        <input type="tel" id="customerPhone" class="form-control" placeholder="07XX XXX XXX" pattern="07[0-9]{8}" maxlength="10" required>
                        <small class="text-muted">Format: 07XXXXXXXX</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="deliveryAddress">Delivery Address *</label>
                        <textarea id="deliveryAddress" class="form-control" rows="3" placeholder="House number, street, apartment, landmarks..." required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label>Select Delivery Location *</label>
                        <div class="delivery-options" id="deliveryOptions">
                            <!-- Delivery options will be dynamically inserted -->
                        </div>
                    </div>
                    
                    <div class="checkout-footer">
                        <button class="btn btn-secondary" id="cancelCheckout">Cancel</button>
                        <button class="btn btn-primary" id="nextToPayment">Next: Payment <i class="fas fa-arrow-right"></i></button>
                    </div>
                </div>
                
                <!-- Step 2: Payment -->
                <div class="checkout-step" id="step2">
                    <h3 class="step-title">Payment Method</h3>
                    
                    <div class="order-summary-small">
                        <div class="total-row">
                            <span>Subtotal</span>
                            <span id="checkoutSubtotal">KES 0</span>
                        </div>
                        <div class="total-row">
                            <span>Delivery Fee</span>
                            <span id="checkoutDeliveryFee">KES 0</span>
                        </div>
                        <div class="total-row">
                            <span>Total</span>
                            <span id="checkoutTotal" style="color: var(--primary); font-size: 1.2rem; font-weight: 700;">KES 0</span>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="deliveryLocationConfirm">Delivery to:</label>
                        <input type="text" id="deliveryLocationConfirm" class="form-control" readonly>
                    </div>
                    
                    <div class="form-group">
                        <label>Payment Method *</label>
                        <div style="display: flex; gap: 10px;">
                            <button class="btn btn-primary" id="mpesaPaymentBtn" style="flex: 1;">
                                <i class="fas fa-mobile-alt"></i> Pay with M-Pesa
                            </button>
                            <button class="btn btn-secondary" id="cashPaymentBtn" style="flex: 1;">
                                <i class="fas fa-money-bill"></i> Pay Cash on Delivery
                            </button>
                        </div>
                    </div>
                    
                    <div class="checkout-footer">
                        <button class="btn btn-secondary" id="backToInfo">
                            <i class="fas fa-arrow-left"></i> Back
                        </button>
                        <button class="btn btn-secondary" id="cancelPayment">Cancel Order</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="main-content">
            <!-- Menu Section -->
            <section class="menu-section">
                <h2 class="section-title"><i class="fas fa-utensils"></i> Our Menu</h2>
                
                <div class="category-filters" id="categoryFilters">
                    <!-- Categories will be loaded dynamically -->
                    <button class="category-btn active" data-category="all">All Items</button>
                </div>
                
                <div class="menu-grid" id="menuGrid">
                    <!-- Menu items will be dynamically inserted -->
                    <div class="menu-loading" id="menuLoading">
                        <div class="spinner"></div>
                        <p>Loading menu...</p>
                    </div>
                </div>
            </section>
            
            <!-- Order Summary -->
            <aside class="order-summary">
                <h2 class="section-title"><i class="fas fa-receipt"></i> Your Order</h2>
                
                <div class="order-items" id="orderItems">
                    <div class="empty-cart" id="emptyCart">
                        <i class="fas fa-shopping-basket"></i>
                        <p>Your cart is empty</p>
                        <p>Add items from the menu to get started</p>
                    </div>
                </div>
                
                <div class="order-totals" id="orderTotals" style="display: none;">
                    <div class="total-row">
                        <span>Subtotal</span>
                        <span id="subtotal">KES 0</span>
                    </div>
                    <div class="total-row">
                        <span>Delivery Fee</span>
                        <span id="deliveryFee">KES 0</span>
                    </div>
                    <div class="total-row">
                        <span>Total</span>
                        <span id="total">KES 0</span>
                    </div>
                </div>
                
                <div class="cart-actions-main" id="cartActions" style="display: none;">
                    <button class="empty-cart-main-btn" id="emptyCartBtn">
                        <i class="fas fa-trash"></i> Empty Cart
                    </button>
                    <button class="checkout-btn" id="checkoutBtn">
                        <i class="fas fa-lock"></i> Checkout Now
                    </button>
                </div>
            </aside>
        </div>
    </div>
    
    <!-- Floating Cart -->
    <div class="floating-cart" id="floatingCart">
        <div class="cart-summary">
            <div class="cart-summary-item">
                <span class="cart-summary-label">Items</span>
                <span class="cart-summary-value" id="floatingItemCount">0</span>
            </div>
            <div class="cart-summary-item">
                <span class="cart-summary-label">Subtotal</span>
                <span class="cart-summary-value" id="floatingSubtotal">KES 0</span>
            </div>
            <div class="cart-summary-item">
                <span class="cart-summary-label">Delivery</span>
                <span class="cart-summary-value" id="floatingDelivery">KES 0</span>
            </div>
            <div class="cart-summary-item">
                <span class="cart-summary-label">Total</span>
                <span class="cart-summary-value" id="floatingTotal">KES 0</span>
            </div>
        </div>
        <div class="cart-actions">
            <button class="empty-cart-btn" id="floatingEmptyCartBtn">
                <i class="fas fa-trash"></i> Empty
            </button>
            <button class="floating-checkout-btn" id="floatingCheckoutBtn">
                <i class="fas fa-lock"></i> Checkout
            </button>
        </div>
    </div>
    
    <!-- Footer -->
    <footer style="background: var(--dark); color: white; padding: 30px 0; margin-top: 50px;">
        <div class="container" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 20px;">
            <div class="footer-logo" style="display: flex; align-items: center; gap: 15px;">
                <img src="https://aromacafe.co.ke/blaqpremium/images/sushilogo.png" 
                     alt="SUSHI-BAR Logo" 
                     style="height: 60px; width: auto;">
            </div>
            <div class="contact-info">
                <div style="margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> <span>Mombasa, Kenya</span></div>
                <div style="margin-bottom: 10px;"><i class="fas fa-phone"></i> <span>+254 790 644 644</span></div>
                <div><i class="fab fa-whatsapp"></i> <span>+254 790 644 644</span></div>
            </div>
            <div style="text-align: center; width: 100%; margin-top: 20px;">
                <p>© 2025 SUSHI-BAR. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <script>
        // Global variables
        let menuItems = [];
        let categories = [];
        let deliveryLocations = [];
        let cart = JSON.parse(localStorage.getItem('sushi_bar_cart')) || [];
        let selectedDelivery = null;
        let isLoading = false;
        
        // DOM Elements
        const menuGrid = document.getElementById('menuGrid');
        const menuLoading = document.getElementById('menuLoading');
        const categoryFilters = document.getElementById('categoryFilters');
        const orderItems = document.getElementById('orderItems');
        const emptyCart = document.getElementById('emptyCart');
        const orderTotals = document.getElementById('orderTotals');
        const cartActions = document.getElementById('cartActions');
        const checkoutBtn = document.getElementById('checkoutBtn');
        const emptyCartBtn = document.getElementById('emptyCartBtn');
        const cartCount = document.getElementById('cartCount');
        const floatingCart = document.getElementById('floatingCart');
        const floatingCheckoutBtn = document.getElementById('floatingCheckoutBtn');
        const floatingEmptyCartBtn = document.getElementById('floatingEmptyCartBtn');
        const checkoutModal = document.getElementById('checkoutModal');
        const closeCheckout = document.getElementById('closeCheckout');
        const cancelCheckout = document.getElementById('cancelCheckout');
        const cancelPayment = document.getElementById('cancelPayment');
        const deliveryOptions = document.getElementById('deliveryOptions');
        const nextToPayment = document.getElementById('nextToPayment');
        const backToInfo = document.getElementById('backToInfo');
        const mpesaPaymentBtn = document.getElementById('mpesaPaymentBtn');
        const cashPaymentBtn = document.getElementById('cashPaymentBtn');
        const loadingOverlay = document.getElementById('loadingOverlay');
        const loadingMessage = document.getElementById('loadingMessage');
        
        // Fallback menu data in case API fails
        const fallbackMenuItems = [
            {
                id: 2,
                name: "TEKKA MAKI",
                category: "sushi",
                description: "A timeless sushi classic — fresh, ruby-red tuna wrapped in seasoned sushi rice and crisp nori seaweed. Clean, delicate, and full of ocean flavor",
                price: 300
            },
            {
                id: 3,
                name: "AVOCADO MAKI",
                category: "sushi",
                description: "Smooth, creamy avocado wrapped in perfectly seasoned sushi rice and crisp nori seaweed.",
                price: 800
            },
            {
                id: 4,
                name: "VASAI ROLL",
                category: "sushi",
                description: "A vibrant medley of fresh seasonal vegetables — crisp cucumber, sweet bell pepper, creamy avocado, all wrapped in fluffy sushi rice and nori seaweed",
                price: 800
            },
            {
                id: 5,
                name: "PANKO SUSHI",
                category: "sushi",
                description: "A delicious twist on traditional sushi — fresh fillings wrapped in rice and nori, coated in golden panko breadcrumbs, and lightly fried to a perfect crunch.",
                price: 1000
            },
            {
                id: 6,
                name: "TORI MAKI",
                category: "sushi",
                description: "A delightful fusion of tender grilled chicken, crisp cucumber, and creamy avocado, all wrapped in perfectly seasoned sushi rice and nori seaweed",
                price: 1000
            },
            {
                id: 7,
                name: "EBI TEMPURA",
                category: "sushi",
                description: "Juicy prawns coated in a light, crispy tempura batter and fried to golden perfection. Served with a classic Japanese dipping sauce.",
                price: 1000
            },
            {
                id: 8,
                name: "SAKE MAKI",
                category: "sushi",
                description: "Fresh, tender salmon rolled in fragrant sushi rice and crisp nori seaweed. Smooth and buttery with a hint of ocean sweetness",
                price: 1000
            },
            {
                id: 9,
                name: "Vegetable Fried Noodles",
                category: "noodles",
                description: "A colorful mix of fresh seasonal vegetables stir-fried with springy rice noodles in a light, flavorful sauce — simple, wholesome, and delicious.",
                price: 700
            },
            {
                id: 10,
                name: "Chicken Chow Mein",
                category: "noodles",
                description: "Stir-fried rice noodles tossed with tender chicken, crisp vegetables, and a savory wok-fresh flavor that captures the essence of Chinese comfort food.",
                price: 800
            },
            {
                id: 11,
                name: "Beef Teriyaki Noodles",
                category: "noodles",
                description: "Golden rice noodles wok-tossed with juicy beef strips and fresh vegetables in a rich, aromatic sauce — hearty, flavorful, and satisfying.",
                price: 800
            },
            {
                id: 12,
                name: "Yasai Yaki Udon – Vegie Fried Udon",
                category: "noodles",
                description: "A delightful vegetarian favorite — thick udon noodles tossed with fresh, crisp vegetables and a light soy-based seasoning for a wholesome, satisfying meal.",
                price: 700
            },
            {
                id: 13,
                name: "Tori Yaki Udon – Chicken Fried Udon",
                category: "noodles",
                description: "Thick, chewy wheat noodles stir-fried with tender chicken and fresh vegetables in a savory soy glaze — comforting and full of Japanese flavor.",
                price: 800
            },
            {
                id: 14,
                name: "Gyuniku Yaki Udon – Beef Fried Udon",
                category: "noodles",
                description: "Soft, silky udon noodles wok-fried with marinated beef strips and vegetables, bringing together rich umami flavor and hearty texture in every bite.",
                price: 900
            },
            {
                id: 15,
                name: "Sticky Chinese Chicken Wings",
                category: "wings",
                description: "A sweet and tangy honey glazed finger licking savory wings.",
                price: 750
            },
            {
                id: 16,
                name: "Fried Chicken Wings",
                category: "wings",
                description: "Well seasoned Juicy chicken wings deep fried until golden brown.",
                price: 750
            },
            {
                id: 17,
                name: "Air Fryer Chicken Nuggets",
                category: "wings",
                description: "Crispy golden chicken wings cooked in an air fryer for a healthy crunch.",
                price: 750
            },
            {
                id: 18,
                name: "Chicken wings roll pop",
                category: "wings",
                description: "Crispy perfection! Tender chicken rolled, dipped in light tempura batter, coated with golden panko crumbs and fried to a satisfying crunch.",
                price: 750
            }
        ];

        const fallbackDeliveryLocations = [
            { name: "Bamburi", cost: 0 },
            { name: "Utange", cost: 100 },
            { name: "Kiembeni", cost: 150 },
            { name: "Shanzu", cost: 200 },
            { name: "Mtwapa", cost: 300 },
            { name: "Serena", cost: 300 },
            { name: "Nyali", cost: 300 },
            { name: "Kisauni", cost: 150 },
            { name: "Tudor", cost: 400 },
            { name: "Mombasa Island", cost: 500 }
        ];

        // Utility functions
        function showNotification(message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = type === 'success' ? 'success-message slide-up' : 'error-message slide-up';
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                <span>${message}</span>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }

        function showLoading(message = 'Processing...') {
            isLoading = true;
            loadingMessage.textContent = message;
            loadingOverlay.style.display = 'flex';
        }

        function hideLoading() {
            isLoading = false;
            loadingOverlay.style.display = 'none';
        }

        // Mobile optimization for floating cart
        function updateCartForMobile() {
            if (window.innerWidth <= 768) {
                // On mobile, ensure cart doesn't take too much space
                const floatingCart = document.getElementById('floatingCart');
                const cartContentHeight = floatingCart.scrollHeight;
                const screenHeight = window.innerHeight;
                
                // If cart content is more than 40% of screen, add scroll indicator
                if (cartContentHeight > screenHeight * 0.4) {
                    floatingCart.style.maxHeight = '33vh'; // One third of viewport
                    floatingCart.style.overflowY = 'auto';
                }
            } else {
                // Reset styles for larger screens
                floatingCart.style.maxHeight = 'none';
                floatingCart.style.overflowY = 'visible';
            }
        }

        // Fetch menu data from API
        async function fetchMenuData() {
            try {
                menuLoading.style.display = 'block';
                
                const response = await fetch('api/menu.php?v=' + Date.now());
                const data = await response.json();
                
                if (data.success) {
                    menuItems = data.menu_items || [];
                    categories = data.categories || [];
                    deliveryLocations = data.delivery_locations || [];
                    
                    // Update category filters
                    updateCategoryFilters();
                    
                    // Initialize menu
                    initMenu();
                    
                    // Initialize delivery options
                    initDeliveryOptions();
                    
                    showNotification('Menu loaded successfully');
                } else {
                    throw new Error(data.message || 'Failed to load menu');
                }
            } catch (error) {
                console.error('Error loading menu:', error);
                
                // Use fallback data
                menuItems = fallbackMenuItems;
                categories = ['sushi', 'noodles', 'wings'];
                deliveryLocations = fallbackDeliveryLocations;
                
                updateCategoryFilters();
                initMenu();
                initDeliveryOptions();
                
                showNotification('Using offline menu data', 'error');
            } finally {
                menuLoading.style.display = 'none';
            }
        }

        // Update category filters dynamically
        function updateCategoryFilters() {
            // Clear existing filters (keep "All Items")
            const allItemsBtn = categoryFilters.querySelector('[data-category="all"]');
            categoryFilters.innerHTML = '';
            
            if (allItemsBtn) {
                categoryFilters.appendChild(allItemsBtn);
            }
            
            // Add dynamic category filters
            categories.forEach(category => {
                const btn = document.createElement('button');
                btn.className = 'category-btn';
                btn.dataset.category = category.toLowerCase();
                btn.textContent = category.charAt(0).toUpperCase() + category.slice(1);
                btn.addEventListener('click', () => {
                    filterMenu(category.toLowerCase());
                });
                categoryFilters.appendChild(btn);
            });
        }

        // Initialize menu display
        function initMenu() {
            menuGrid.innerHTML = '';
            
            if (menuItems.length === 0) {
                menuGrid.innerHTML = `
                    <div class="empty-menu" style="grid-column: 1/-1; text-align: center; padding: 40px;">
                        <i class="fas fa-utensils fa-3x text-muted mb-3"></i>
                        <h4>No menu items available</h4>
                        <p class="text-muted">Please check back later</p>
                    </div>
                `;
                return;
            }
            
            menuItems.forEach(item => {
                const menuItemElement = document.createElement('div');
                menuItemElement.className = 'menu-item fade-in';
                menuItemElement.dataset.category = item.category ? item.category.toLowerCase() : 'uncategorized';
                
                // Determine icon based on category
                let icon = 'fas fa-utensils';
                const categoryLower = item.category ? item.category.toLowerCase() : '';
                
                if (categoryLower.includes('sushi')) {
                    icon = 'fas fa-fish';
                } else if (categoryLower.includes('noodle')) {
                    icon = 'fas fa-utensil-spoon';
                } else if (categoryLower.includes('wing') || categoryLower.includes('chicken')) {
                    icon = 'fas fa-drumstick-bite';
                }
                
                // Handle image
                let imageHtml = '';
                if (item.image_url) {
                    imageHtml = `
                        <img src="${item.image_url}" 
                             alt="${item.name}" 
                             onerror="this.onerror=null; this.parentElement.innerHTML='<i class=\'${icon}\'></i>'">
                    `;
                } else {
                    imageHtml = `<i class="${icon}"></i>`;
                }
                
                menuItemElement.innerHTML = `
                    <div class="item-image">
                        ${imageHtml}
                        ${item.category ? `<div class="item-category">${item.category.toUpperCase()}</div>` : ''}
                    </div>
                    <div class="item-details">
                        <h3 class="item-name">${item.name}</h3>
                        <p class="item-description">${item.description || ''}</p>
                        <div class="item-footer">
                            <div class="item-price">KES ${parseFloat(item.price).toFixed(2)}</div>
                            <button class="add-to-cart" data-id="${item.id}">
                                <i class="fas fa-plus"></i> Add to Cart
                            </button>
                        </div>
                    </div>
                `;
                
                menuGrid.appendChild(menuItemElement);
            });
            
            // Add event listeners to Add to Cart buttons
            document.querySelectorAll('.add-to-cart').forEach(button => {
                button.addEventListener('click', addToCart);
            });
        }

        // Filter menu by category
        function filterMenu(category) {
            const menuItems = document.querySelectorAll('.menu-item');
            
            menuItems.forEach(item => {
                if (category === 'all' || item.dataset.category === category.toLowerCase()) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
            
            // Update active button
            document.querySelectorAll('.category-btn').forEach(btn => {
                if (btn.dataset.category === category) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
        }

        // Initialize delivery options
        function initDeliveryOptions() {
            deliveryOptions.innerHTML = '';
            
            deliveryLocations.forEach(location => {
                const optionElement = document.createElement('div');
                optionElement.className = 'delivery-option';
                optionElement.dataset.name = location.name;
                optionElement.dataset.cost = location.cost;
                
                const costText = location.cost === 0 ? 'FREE' : `KES ${location.cost}`;
                const costClass = location.cost === 0 ? 'free' : '';
                
                optionElement.innerHTML = `
                    <div class="location-name">${location.name}</div>
                    <div class="delivery-cost ${costClass}">${costText}</div>
                `;
                
                optionElement.addEventListener('click', () => {
                    document.querySelectorAll('.delivery-option').forEach(option => {
                        option.classList.remove('selected');
                    });
                    
                    optionElement.classList.add('selected');
                    selectedDelivery = location;
                    updateCart();
                    updateCheckoutSummary();
                });
                
                deliveryOptions.appendChild(optionElement);
            });
            
            // Select first location by default
            if (deliveryLocations.length > 0) {
                const firstOption = deliveryOptions.querySelector('.delivery-option');
                if (firstOption) {
                    firstOption.click();
                }
            }
        }

        // Cart functions
        function addToCart(e) {
            if (isLoading) return;
            
            const itemId = parseInt(e.currentTarget.dataset.id);
            const item = menuItems.find(menuItem => menuItem.id === itemId);
            
            if (!item) {
                showNotification('Item not found!', 'error');
                return;
            }
            
            const existingItem = cart.find(cartItem => cartItem.id === itemId);
            
            if (existingItem) {
                existingItem.quantity += 1;
            } else {
                cart.push({
                    ...item,
                    quantity: 1
                });
            }
            
            saveCart();
            updateCart();
            showNotification(`${item.name} added to cart!`);
        }

        function saveCart() {
            localStorage.setItem('sushi_bar_cart', JSON.stringify(cart));
        }

        function emptyCartFunc() {
            if (cart.length === 0 || isLoading) return;
            
            if (confirm('Are you sure you want to empty your cart?')) {
                cart = [];
                selectedDelivery = null;
                saveCart();
                updateCart();
                showNotification('Cart emptied successfully');
            }
        }

        function updateCart() {
            // Update cart count
            const totalItems = cart.reduce((total, item) => total + item.quantity, 0);
            cartCount.textContent = totalItems;
            
            // Calculate subtotal
            const subtotal = cart.reduce((total, item) => total + (item.price * item.quantity), 0);
            
            // Calculate delivery fee
            const deliveryFee = selectedDelivery ? selectedDelivery.cost : 0;
            const total = subtotal + deliveryFee;
            
            // Update cart display
            updateCartDisplay(totalItems, subtotal, deliveryFee, total);
            
            // Mobile optimization
            updateCartForMobile();
            
            // If cart is empty
            if (cart.length === 0) {
                emptyCart.style.display = 'block';
                orderTotals.style.display = 'none';
                cartActions.style.display = 'none';
                floatingCart.classList.remove('visible');
                return;
            }
            
            // Hide empty cart message
            emptyCart.style.display = 'none';
            
            // Update order items
            orderItems.innerHTML = '';
            
            cart.forEach(item => {
                const orderItemElement = document.createElement('div');
                orderItemElement.className = 'order-item';
                
                orderItemElement.innerHTML = `
                    <div class="order-item-name">${item.name}</div>
                    <div class="order-item-controls">
                        <button class="quantity-btn decrease" data-id="${item.id}">-</button>
                        <div class="item-quantity">${item.quantity}</div>
                        <button class="quantity-btn increase" data-id="${item.id}">+</button>
                        <div class="order-item-price">KES ${(item.price * item.quantity).toFixed(2)}</div>
                        <button class="remove-item" data-id="${item.id}" title="Remove item">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                `;
                
                orderItems.appendChild(orderItemElement);
            });
            
            // Add event listeners
            document.querySelectorAll('.decrease').forEach(button => {
                button.addEventListener('click', decreaseQuantity);
            });
            
            document.querySelectorAll('.increase').forEach(button => {
                button.addEventListener('click', increaseQuantity);
            });
            
            document.querySelectorAll('.remove-item').forEach(button => {
                button.addEventListener('click', removeItem);
            });
            
            // Show totals and checkout button
            orderTotals.style.display = 'block';
            cartActions.style.display = 'flex';
            floatingCart.classList.add('visible');
        }

        function updateCartDisplay(itemCount, subtotal, deliveryFee, total) {
            // Update sidebar
            document.getElementById('subtotal').textContent = `KES ${subtotal.toFixed(2)}`;
            document.getElementById('deliveryFee').textContent = `KES ${deliveryFee.toFixed(2)}`;
            document.getElementById('total').textContent = `KES ${total.toFixed(2)}`;
            
            // Update floating cart
            document.getElementById('floatingItemCount').textContent = itemCount;
            document.getElementById('floatingSubtotal').textContent = `KES ${subtotal.toFixed(2)}`;
            document.getElementById('floatingDelivery').textContent = `KES ${deliveryFee.toFixed(2)}`;
            document.getElementById('floatingTotal').textContent = `KES ${total.toFixed(2)}`;
            
            // Update checkout summary
            updateCheckoutSummary();
        }

        function updateCheckoutSummary() {
            const subtotal = cart.reduce((total, item) => total + (item.price * item.quantity), 0);
            const deliveryFee = selectedDelivery ? selectedDelivery.cost : 0;
            const total = subtotal + deliveryFee;
            
            document.getElementById('checkoutSubtotal').textContent = `KES ${subtotal.toFixed(2)}`;
            document.getElementById('checkoutDeliveryFee').textContent = `KES ${deliveryFee.toFixed(2)}`;
            document.getElementById('checkoutTotal').textContent = `KES ${total.toFixed(2)}`;
            
            if (selectedDelivery) {
                document.getElementById('deliveryLocationConfirm').value = 
                    `${selectedDelivery.name} (${deliveryFee === 0 ? 'Free' : `KES ${deliveryFee}`})`;
            }
        }

        function decreaseQuantity(e) {
            if (isLoading) return;
            
            const itemId = parseInt(e.currentTarget.dataset.id);
            const item = cart.find(item => item.id === itemId);
            
            if (item.quantity > 1) {
                item.quantity -= 1;
            } else {
                cart = cart.filter(cartItem => cartItem.id !== itemId);
            }
            
            saveCart();
            updateCart();
        }

        function increaseQuantity(e) {
            if (isLoading) return;
            
            const itemId = parseInt(e.currentTarget.dataset.id);
            const item = cart.find(item => item.id === itemId);
            
            item.quantity += 1;
            saveCart();
            updateCart();
        }

        function removeItem(e) {
            if (isLoading) return;
            
            const itemId = parseInt(e.currentTarget.dataset.id);
            cart = cart.filter(item => item.id !== itemId);
            saveCart();
            updateCart();
            showNotification('Item removed from cart');
        }

        // Checkout functions
        function showCheckoutModal() {
            if (cart.length === 0 || isLoading) {
                showNotification('Your cart is empty!', 'error');
                return;
            }
            
            // Reset modal state
            document.getElementById('step1').classList.add('active');
            document.getElementById('step2').classList.remove('active');
            document.getElementById('customerName').value = '';
            document.getElementById('customerPhone').value = '';
            document.getElementById('deliveryAddress').value = '';
            
            // Initialize delivery options if needed
            if (deliveryOptions.children.length === 0) {
                initDeliveryOptions();
            }
            
            // Update checkout summary
            updateCheckoutSummary();
            
            // Show modal
            checkoutModal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }

        function closeCheckoutModal() {
            checkoutModal.classList.remove('active');
            document.body.style.overflow = 'auto';
        }

        function validateCheckoutStep1() {
            const customerName = document.getElementById('customerName').value.trim();
            const customerPhone = document.getElementById('customerPhone').value.trim();
            const deliveryAddress = document.getElementById('deliveryAddress').value.trim();
            
            if (!customerName) {
                showNotification('Please enter your name', 'error');
                return false;
            }
            
            if (!customerPhone.match(/^07[0-9]{8}$/)) {
                showNotification('Please enter a valid Kenyan phone number (07XXXXXXXX)', 'error');
                return false;
            }
            
            if (!deliveryAddress) {
                showNotification('Please enter your delivery address', 'error');
                return false;
            }
            
            if (!selectedDelivery) {
                showNotification('Please select a delivery location', 'error');
                return false;
            }
            
            return true;
        }

        // Payment processing functions
        async function processMpesaPayment() {
            if (!validateCheckoutStep1() || isLoading) return;
            
            const customerName = document.getElementById('customerName').value.trim();
            const customerPhone = document.getElementById('customerPhone').value.trim();
            const deliveryAddress = document.getElementById('deliveryAddress').value.trim();
            
            // Calculate totals
            const subtotal = cart.reduce((total, item) => total + (item.price * item.quantity), 0);
            const deliveryFee = selectedDelivery.cost;
            const total = subtotal + deliveryFee;
            
            // Prepare order data
            const orderData = {
                customer_name: customerName,
                customer_phone: customerPhone,
                customer_address: deliveryAddress + ', ' + selectedDelivery.name,
                payment_method: 'mpesa',
                cart: cart.map(item => ({
                    name: item.name,
                    price: item.price,
                    quantity: item.quantity
                })),
                total_amount: total,
                delivery_location: selectedDelivery.name,
                delivery_fee: deliveryFee
            };
            
            showLoading('Initiating M-Pesa payment...');
            
            try {
                const response = await fetch('process_order.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(orderData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    if (result.payment_initiated) {
                        // Save order info for polling
                        localStorage.setItem('last_order', JSON.stringify({
                            order_id: result.order_id,
                            order_number: result.order_number,
                            checkout_id: result.checkout_id,
                            total: total
                        }));
                        
                        showLoading('✓ Payment request sent to your phone. Please check and enter your M-Pesa PIN.');
                        
                        // Start polling for payment status
                        setTimeout(() => {
                            checkMpesaPaymentStatus(result.order_id, result.checkout_id);
                        }, 2000);
                    } else {
                        hideLoading();
                        showNotification('Order recorded: ' + result.message);
                    }
                } else {
                    hideLoading();
                    showNotification('Payment failed: ' + (result.message || 'Please try again'), 'error');
                }
            } catch (error) {
                hideLoading();
                console.error('Order error:', error);
                showNotification('Network error: ' + error.message, 'error');
            }
        }

        async function checkMpesaPaymentStatus(orderId, checkoutId) {
            let attempts = 0;
            const maxAttempts = 30;
            
            const pollInterval = setInterval(async () => {
                attempts++;
                
                try {
                    const response = await fetch(`check_payment_status.php?order_id=${orderId}&checkout_id=${checkoutId}`);
                    const result = await response.json();
                    
                    if (result.payment_status === 'completed') {
                        clearInterval(pollInterval);
                        hideLoading();
                        closeCheckoutModal();
                        
                        // Clear cart
                        cart = [];
                        selectedDelivery = null;
                        saveCart();
                        updateCart();
                        
                        // Redirect to success page
                        window.location.href = `order_success.php?order_id=${orderId}&order_number=${result.order_number}`;
                        
                    } else if (result.payment_status === 'failed' || attempts >= maxAttempts) {
                        clearInterval(pollInterval);
                        hideLoading();
                        
                        if (attempts >= maxAttempts) {
                            showNotification('Payment is taking longer than expected. Your order has been recorded.', 'error');
                        } else {
                            showNotification('Payment failed. Please try again or contact support.', 'error');
                        }
                    } else {
                        // Still pending
                        const messages = [
                            'Waiting for payment confirmation...',
                            'Please complete payment on your phone...',
                            'Checking payment status...',
                            'Payment is being processed...'
                        ];
                        const messageIndex = Math.floor(attempts / 3) % messages.length;
                        showLoading(messages[messageIndex]);
                    }
                } catch (error) {
                    console.error('Status check error:', error);
                    
                    if (attempts >= maxAttempts) {
                        clearInterval(pollInterval);
                        hideLoading();
                        showNotification('Unable to verify payment status.', 'error');
                    }
                }
            }, 5000);
        }

        async function processCashPayment() {
            if (!validateCheckoutStep1() || isLoading) return;
            
            const customerName = document.getElementById('customerName').value.trim();
            const customerPhone = document.getElementById('customerPhone').value.trim();
            const deliveryAddress = document.getElementById('deliveryAddress').value.trim();
            
            // Calculate totals
            const subtotal = cart.reduce((total, item) => total + (item.price * item.quantity), 0);
            const deliveryFee = selectedDelivery.cost;
            const total = subtotal + deliveryFee;
            
            // Prepare order data
            const orderData = {
                customer_name: customerName,
                customer_phone: customerPhone,
                customer_address: deliveryAddress + ', ' + selectedDelivery.name,
                payment_method: 'cash',
                cart: cart.map(item => ({
                    name: item.name,
                    price: item.price,
                    quantity: item.quantity
                })),
                total_amount: total,
                delivery_location: selectedDelivery.name,
                delivery_fee: deliveryFee
            };
            
            showLoading('Processing cash order...');
            
            try {
                const response = await fetch('process_order.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(orderData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    hideLoading();
                    closeCheckoutModal();
                    
                    // Clear cart
                    cart = [];
                    selectedDelivery = null;
                    saveCart();
                    updateCart();
                    
                    showNotification('Cash order placed successfully!');
                    window.location.href = `order_success.php?order_id=${result.order_id}&order_number=${result.order_number}`;
                } else {
                    hideLoading();
                    showNotification('Order failed: ' + (result.message || 'Please try again'), 'error');
                }
            } catch (error) {
                hideLoading();
                console.error('Order error:', error);
                showNotification('Network error: ' + error.message, 'error');
            }
        }

        // Initialize the application
        function init() {
            // Fetch menu data first
            fetchMenuData().then(() => {
                updateCart();
                
                // Event listeners for category filters
                document.querySelectorAll('.category-btn').forEach(btn => {
                    btn.addEventListener('click', () => {
                        filterMenu(btn.dataset.category);
                    });
                });
                
                // Event listeners for checkout buttons
                checkoutBtn.addEventListener('click', showCheckoutModal);
                floatingCheckoutBtn.addEventListener('click', showCheckoutModal);
                
                // Event listeners for empty cart buttons
                emptyCartBtn.addEventListener('click', emptyCartFunc);
                floatingEmptyCartBtn.addEventListener('click', emptyCartFunc);
                
                // Event listeners for checkout modal
                closeCheckout.addEventListener('click', closeCheckoutModal);
                cancelCheckout.addEventListener('click', closeCheckoutModal);
                cancelPayment.addEventListener('click', closeCheckoutModal);
                
                // Event listener for next button in checkout
                nextToPayment.addEventListener('click', () => {
                    if (validateCheckoutStep1()) {
                        document.getElementById('step1').classList.remove('active');
                        document.getElementById('step2').classList.add('active');
                    }
                });
                
                // Event listener for back button
                backToInfo.addEventListener('click', () => {
                    document.getElementById('step2').classList.remove('active');
                    document.getElementById('step1').classList.add('active');
                });
                
                // Event listeners for payment buttons
                mpesaPaymentBtn.addEventListener('click', processMpesaPayment);
                cashPaymentBtn.addEventListener('click', processCashPayment);
                
                // Event listener for cart icon
                const cartIcon = document.getElementById('cartIcon');
                cartIcon.addEventListener('click', () => {
                    document.querySelector('.order-summary').scrollIntoView({ 
                        behavior: 'smooth' 
                    });
                });
                
                // Event listener for window resize
                window.addEventListener('resize', updateCartForMobile);
            });
        }

        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', init);
        
        // Auto-refresh menu every 5 minutes
        setInterval(() => {
            if (!isLoading) {
                fetchMenuData();
            }
        }, 300000); // 5 minutes
    </script>
</body>
</html>