<?php
// mpesa/callback.php - UPDATED WITH CORRECT PATHS

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../callback_error.log');

// Get the raw callback data
$callbackData = file_get_contents('php://input');

// Log the callback for debugging
$logDir = __DIR__ . '/../logs';
if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

$logFile = $logDir . '/mpesa_callback.log';
$timestamp = date('Y-m-d H:i:s');
file_put_contents($logFile, "[" . $timestamp . "] Received: " . $callbackData . "\n", FILE_APPEND);

// Check if callback data is empty
if (empty($callbackData)) {
    file_put_contents($logFile, "[" . $timestamp . "] ERROR: Empty callback data\n", FILE_APPEND);
    echo json_encode([
        'ResultCode' => 1,
        'ResultDesc' => 'Empty callback data'
    ]);
    exit;
}

// Decode the callback data
$data = json_decode($callbackData, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    file_put_contents($logFile, "[" . $timestamp . "] ERROR: Invalid JSON: " . json_last_error_msg() . "\n", FILE_APPEND);
    echo json_encode([
        'ResultCode' => 1,
        'ResultDesc' => 'Invalid JSON data'
    ]);
    exit;
}

file_put_contents($logFile, "[" . $timestamp . "] Decoded: " . print_r($data, true) . "\n", FILE_APPEND);

// Include database config with correct path
require_once __DIR__ . '/../config/database.php';

if (isset($data['Body']['stkCallback'])) {
    $callback = $data['Body']['stkCallback'];
    $resultCode = $callback['ResultCode'];
    $resultDesc = $callback['ResultDesc'];
    $checkoutRequestId = $callback['CheckoutRequestID'];
    $merchantRequestId = $callback['MerchantRequestID'];
    
    file_put_contents($logFile, "[" . $timestamp . "] Processing: ResultCode=$resultCode, CheckoutID=$checkoutRequestId\n", FILE_APPEND);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    if (!$conn) {
        file_put_contents($logFile, "[" . $timestamp . "] ERROR: Database connection failed\n", FILE_APPEND);
        echo json_encode([
            'ResultCode' => 1,
            'ResultDesc' => 'Database connection failed'
        ]);
        exit;
    }
    
    if ($resultCode == 0) {
        // Payment successful
        if (!isset($callback['CallbackMetadata']['Item'])) {
            file_put_contents($logFile, "[" . $timestamp . "] ERROR: No callback metadata\n", FILE_APPEND);
            echo json_encode([
                'ResultCode' => 1,
                'ResultDesc' => 'No callback metadata'
            ]);
            exit;
        }
        
        $callbackMetadata = $callback['CallbackMetadata']['Item'];
        
        $amount = 0;
        $mpesaReceiptNumber = '';
        $transactionDate = '';
        $phoneNumber = '';
        
        foreach ($callbackMetadata as $item) {
            switch ($item['Name']) {
                case 'Amount':
                    $amount = $item['Value'];
                    break;
                case 'MpesaReceiptNumber':
                    $mpesaReceiptNumber = $item['Value'];
                    break;
                case 'TransactionDate':
                    $transactionDate = $item['Value'];
                    break;
                case 'PhoneNumber':
                    $phoneNumber = $item['Value'];
                    break;
            }
        }
        
        // Format transaction date (YYYYMMDDHHMMSS to datetime)
        if (strlen($transactionDate) == 14) {
            $year = substr($transactionDate, 0, 4);
            $month = substr($transactionDate, 4, 2);
            $day = substr($transactionDate, 6, 2);
            $hour = substr($transactionDate, 8, 2);
            $minute = substr($transactionDate, 10, 2);
            $second = substr($transactionDate, 12, 2);
            $formattedDate = "$year-$month-$day $hour:$minute:$second";
        } else {
            $formattedDate = date('Y-m-d H:i:s');
        }
        
        file_put_contents($logFile, "[" . $timestamp . "] Success: Amount=$amount, Receipt=$mpesaReceiptNumber, Phone=$phoneNumber\n", FILE_APPEND);
        
        try {
            // Update transaction log
            $updateTransaction = "UPDATE mpesa_transactions 
                                 SET result_code = :result_code,
                                     result_desc = :result_desc,
                                     amount = :amount,
                                     mpesa_receipt_number = :receipt_number,
                                     transaction_date = :transaction_date,
                                     phone_number = :phone_number,
                                     status = 'success',
                                     updated_at = NOW()
                                 WHERE checkout_request_id = :checkout_id";
            
            $stmt = $conn->prepare($updateTransaction);
            $stmt->bindParam(':result_code', $resultCode, PDO::PARAM_INT);
            $stmt->bindParam(':result_desc', $resultDesc);
            $stmt->bindParam(':amount', $amount);
            $stmt->bindParam(':receipt_number', $mpesaReceiptNumber);
            $stmt->bindParam(':transaction_date', $formattedDate);
            $stmt->bindParam(':phone_number', $phoneNumber);
            $stmt->bindParam(':checkout_id', $checkoutRequestId);
            
            if (!$stmt->execute()) {
                $error = $stmt->errorInfo();
                file_put_contents($logFile, "[" . $timestamp . "] ERROR updating transaction: " . print_r($error, true) . "\n", FILE_APPEND);
            }
            
            // Find and update the corresponding order
            $findOrder = "SELECT id FROM orders WHERE mpesa_checkout_id = :checkout_id";
            $stmt = $conn->prepare($findOrder);
            $stmt->bindParam(':checkout_id', $checkoutRequestId);
            $stmt->execute();
            $order = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($order) {
                $updateOrder = "UPDATE orders 
                               SET status = 'paid',
                                   mpesa_receipt = :receipt_number,
                                   mpesa_transaction_date = :transaction_date,
                                   updated_at = NOW()
                               WHERE id = :order_id";
                
                $stmt = $conn->prepare($updateOrder);
                $stmt->bindParam(':receipt_number', $mpesaReceiptNumber);
                $stmt->bindParam(':transaction_date', $formattedDate);
                $stmt->bindParam(':order_id', $order['id']);
                
                if (!$stmt->execute()) {
                    $error = $stmt->errorInfo();
                    file_put_contents($logFile, "[" . $timestamp . "] ERROR updating order: " . print_r($error, true) . "\n", FILE_APPEND);
                } else {
                    file_put_contents($logFile, "[" . $timestamp . "] SUCCESS: Order {$order['id']} updated to paid\n", FILE_APPEND);
                }
            } else {
                file_put_contents($logFile, "[" . $timestamp . "] WARNING: No order found for checkout_id $checkoutRequestId\n", FILE_APPEND);
            }
            
            // Send success response to Safaricom
            echo json_encode([
                'ResultCode' => 0,
                'ResultDesc' => 'Success'
            ]);
            
        } catch (Exception $e) {
            file_put_contents($logFile, "[" . $timestamp . "] EXCEPTION: " . $e->getMessage() . "\n", FILE_APPEND);
            echo json_encode([
                'ResultCode' => 0, // Still return success to Safaricom
                'ResultDesc' => 'Success (with internal error)'
            ]);
        }
        
    } else {
        // Payment failed
        file_put_contents($logFile, "[" . $timestamp . "] Failed: $resultDesc\n", FILE_APPEND);
        
        try {
            // Update transaction log
            $updateTransaction = "UPDATE mpesa_transactions 
                                 SET result_code = :result_code,
                                     result_desc = :result_desc,
                                     status = 'failed',
                                     updated_at = NOW()
                                 WHERE checkout_request_id = :checkout_id";
            
            $stmt = $conn->prepare($updateTransaction);
            $stmt->bindParam(':result_code', $resultCode, PDO::PARAM_INT);
            $stmt->bindParam(':result_desc', $resultDesc);
            $stmt->bindParam(':checkout_id', $checkoutRequestId);
            
            if (!$stmt->execute()) {
                $error = $stmt->errorInfo();
                file_put_contents($logFile, "[" . $timestamp . "] ERROR updating failed transaction: " . print_r($error, true) . "\n", FILE_APPEND);
            }
            
            // Update order status to cancelled
            $updateOrder = "UPDATE orders 
                           SET status = 'cancelled',
                               updated_at = NOW()
                           WHERE mpesa_checkout_id = :checkout_id";
            
            $stmt = $conn->prepare($updateOrder);
            $stmt->bindParam(':checkout_id', $checkoutRequestId);
            
            if (!$stmt->execute()) {
                $error = $stmt->errorInfo();
                file_put_contents($logFile, "[" . $timestamp . "] ERROR updating order to cancelled: " . print_r($error, true) . "\n", FILE_APPEND);
            }
            
            // Send success response to Safaricom (we still acknowledge receipt)
            echo json_encode([
                'ResultCode' => 0,
                'ResultDesc' => 'Success'
            ]);
            
        } catch (Exception $e) {
            file_put_contents($logFile, "[" . $timestamp . "] EXCEPTION in failed payment: " . $e->getMessage() . "\n", FILE_APPEND);
            echo json_encode([
                'ResultCode' => 0,
                'ResultDesc' => 'Success'
            ]);
        }
    }
} else {
    // Invalid callback
    file_put_contents($logFile, "[" . $timestamp . "] ERROR: Invalid callback format\n", FILE_APPEND);
    echo json_encode([
        'ResultCode' => 1,
        'ResultDesc' => 'Invalid callback format'
    ]);
}
?>