<?php
/**
 * process_order.php
 * Dual outlet order processor with pizza size support
 * UPDATED TO HANDLE PIZZA SIZE VARIATIONS
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error_log.txt');

header('Content-Type: application/json');

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/includes/MpesaService.php';

/* -------------------------------------------------
   METHOD CHECK
------------------------------------------------- */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request']);
    exit;
}

/* -------------------------------------------------
   READ JSON BODY
------------------------------------------------- */
$payload = json_decode(file_get_contents('php://input'), true);

if (!$payload) {
    echo json_encode(['success' => false, 'message' => 'Invalid JSON']);
    exit;
}

/* -------------------------------------------------
   INPUTS
------------------------------------------------- */
$customer_name     = trim($payload['customer_name'] ?? '');
$customer_phone    = trim($payload['customer_phone'] ?? '');
$customer_address  = trim($payload['customer_address'] ?? '');
$delivery_location = trim($payload['delivery_location'] ?? '');
$delivery_fee      = floatval($payload['delivery_fee'] ?? 0);
$payment_method    = strtolower($payload['payment_method'] ?? 'mpesa');
$outlet            = strtolower($payload['outlet'] ?? 'sushi');
$cart              = $payload['cart'] ?? [];
$total_amount      = floatval($payload['total_amount'] ?? 0);

/* -------------------------------------------------
   BASIC VALIDATION
------------------------------------------------- */
if (
    empty($customer_name) ||
    empty($customer_phone) ||
    empty($customer_address) ||
    empty($delivery_location) ||
    empty($cart) ||
    $total_amount <= 0
) {
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}

if (!in_array($outlet, ['sushi', 'milano'])) {
    $outlet = 'sushi';
}

/* -------------------------------------------------
   PHONE NORMALIZATION
------------------------------------------------- */
$phone = preg_replace('/\D/', '', $customer_phone);

if (preg_match('/^07\d{8}$/', $phone)) {
    $phone = '254' . substr($phone, 1);
}

if (!preg_match('/^2547\d{8}$/', $phone)) {
    echo json_encode(['success' => false, 'message' => 'Invalid phone number']);
    exit;
}

/* -------------------------------------------------
   DB CONNECTION
------------------------------------------------- */
try {
    $db = new Database();
    $conn = $db->getConnection();
    $conn->beginTransaction();

    /* -------------------------------------------------
       ORDER NUMBER
    ------------------------------------------------- */
    $prefix = $outlet === 'milano' ? 'MIL-' : 'SUS-';

    $stmt = $conn->prepare("
        SELECT COUNT(*) FROM orders WHERE order_number LIKE :prefix
    ");
    $stmt->execute([':prefix' => $prefix . '%']);
    $count = $stmt->fetchColumn() + 1;

    $order_number = $prefix . str_pad($count, 6, '0', STR_PAD_LEFT);

    /* -------------------------------------------------
       CREATE ORDER
    ------------------------------------------------- */
    $stmt = $conn->prepare("
        INSERT INTO orders (
            order_number,
            customer_name,
            customer_phone,
            customer_address,
            delivery_location,
            delivery_fee,
            payment_method,
            total_amount,
            status,
            outlet,
            created_at
        ) VALUES (
            :order_number,
            :name,
            :phone,
            :address,
            :location,
            :delivery_fee,
            :payment_method,
            :total,
            'pending',
            :outlet,
            NOW()
        )
    ");

    $stmt->execute([
        ':order_number' => $order_number,
        ':name' => $customer_name,
        ':phone' => $phone,
        ':address' => $customer_address,
        ':location' => $delivery_location,
        ':delivery_fee' => $delivery_fee,
        ':payment_method' => $payment_method,
        ':total' => $total_amount,
        ':outlet' => $outlet
    ]);

    $order_id = $conn->lastInsertId();

    /* -------------------------------------------------
       INSERT ORDER ITEMS WITH VARIATION ID SUPPORT
    ------------------------------------------------- */
    // Check if order_items table has variation_id column
    $checkColumnStmt = $conn->query("SHOW COLUMNS FROM order_items LIKE 'variation_id'");
    $hasVariationColumn = $checkColumnStmt->rowCount() > 0;
    
    // Prepare SQL based on whether variation_id column exists
    if ($hasVariationColumn) {
        $itemStmt = $conn->prepare("
            INSERT INTO order_items (
                order_id,
                item_name,
                item_price,
                item_size,
                variation_id,
                quantity,
                subtotal
            ) VALUES (
                :order_id,
                :name,
                :price,
                :size,
                :variation_id,
                :qty,
                :subtotal
            )
        ");
    } else {
        $itemStmt = $conn->prepare("
            INSERT INTO order_items (
                order_id,
                item_name,
                item_price,
                item_size,
                quantity,
                subtotal
            ) VALUES (
                :order_id,
                :name,
                :price,
                :size,
                :qty,
                :subtotal
            )
        ");
    }

    foreach ($cart as $item) {
        $name     = $item['name'] ?? '';
        $qty      = (int)($item['quantity'] ?? 0);
        $size     = $item['size'] ?? null;
        $price    = (float)($item['price'] ?? 0);
        $variation_id = $item['variation_id'] ?? null;
        $subtotal = $price * $qty;

        if ($qty <= 0 || $price <= 0 || empty($name)) {
            throw new Exception('Invalid cart item: ' . json_encode($item));
        }

        // Validate variation_id if provided (for pizzas with size variations)
        if ($variation_id && $outlet === 'milano') {
            // Check if variation exists and belongs to correct product
            $checkVarStmt = $conn->prepare("
                SELECT pv.*, mi.name as product_name 
                FROM product_variations pv
                JOIN menu_items mi ON pv.product_id = mi.id
                WHERE pv.id = :variation_id 
                AND pv.is_active = 1
                AND mi.outlet = :outlet
            ");
            $checkVarStmt->execute([
                ':variation_id' => $variation_id,
                ':outlet' => $outlet
            ]);
            $variation = $checkVarStmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$variation) {
                throw new Exception("Invalid variation ID: {$variation_id} for product");
            }
            
            // Verify the variation matches the item
            if ($variation['size'] !== $size) {
                throw new Exception("Size mismatch for variation: {$variation_id}");
            }
        }

        // Execute with appropriate parameters
        $params = [
            ':order_id' => $order_id,
            ':name'     => $name,
            ':price'    => $price,
            ':size'     => $size,
            ':qty'      => $qty,
            ':subtotal' => $subtotal
        ];
        
        if ($hasVariationColumn) {
            $params[':variation_id'] = $variation_id;
        }

        $itemStmt->execute($params);
    }

    /* -------------------------------------------------
       UPDATE DAILY SALES SUMMARY
    ------------------------------------------------- */
    try {
        $today = date('Y-m-d');
        
        // Check if record exists for today
        $checkStmt = $conn->prepare("
            SELECT id FROM daily_sales_summary 
            WHERE date = :today AND outlet = :outlet
        ");
        $checkStmt->execute([':today' => $today, ':outlet' => $outlet]);
        $existing = $checkStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing) {
            // Update existing record
            $updateStmt = $conn->prepare("
                UPDATE daily_sales_summary 
                SET 
                    total_orders = total_orders + 1,
                    total_revenue = total_revenue + :revenue,
                    updated_at = NOW()
                WHERE id = :id
            ");
            $updateStmt->execute([
                ':revenue' => $total_amount,
                ':id' => $existing['id']
            ]);
        } else {
            // Insert new record
            $insertStmt = $conn->prepare("
                INSERT INTO daily_sales_summary (
                    date,
                    outlet,
                    total_orders,
                    total_revenue,
                    created_at
                ) VALUES (
                    :date,
                    :outlet,
                    1,
                    :revenue,
                    NOW()
                )
            ");
            $insertStmt->execute([
                ':date' => $today,
                ':outlet' => $outlet,
                ':revenue' => $total_amount
            ]);
        }
    } catch (Exception $e) {
        // Log but don't fail the order if sales summary update fails
        error_log('Daily sales summary update failed: ' . $e->getMessage());
    }

    $conn->commit();

    /* -------------------------------------------------
       SEND ORDER CONFIRMATION (Optional)
    ------------------------------------------------- */
    try {
        // You can add WhatsApp/SMS notification here
        // Example: sendWhatsAppNotification($order_number, $customer_name, $customer_phone, $total_amount);
    } catch (Exception $e) {
        // Don't fail order if notification fails
        error_log('Notification failed: ' . $e->getMessage());
    }

} catch (Exception $e) {
    if (isset($conn) && $conn->inTransaction()) {
        $conn->rollBack();
    }

    error_log('ORDER PROCESSING ERROR: ' . $e->getMessage() . PHP_EOL . 'Payload: ' . json_encode($payload));

    echo json_encode([
        'success' => false,
        'message' => 'Order processing failed: ' . $e->getMessage()
    ]);
    exit;
}

/* -------------------------------------------------
   PAYMENT PROCESSING
------------------------------------------------- */
if ($payment_method === 'mpesa') {
    try {
        $mpesa = new MpesaService();

        $description = ($outlet === 'milano')
            ? "MILANO Pizza Order #{$order_number}"
            : "SUSHI-BAR Order #{$order_number}";

        $response = $mpesa->initiateSTKPush(
            $phone,
            $total_amount,
            $order_id,
            $description
        );

        if ($response['success']) {
            // Update order with checkout ID
            $stmt = $conn->prepare("
                UPDATE orders
                SET 
                    mpesa_checkout_id = :checkout,
                    status = 'processing',
                    updated_at = NOW()
                WHERE id = :id
            ");
            $stmt->execute([
                ':checkout' => $response['checkout_id'],
                ':id' => $order_id
            ]);

            // Log M-Pesa transaction
            try {
                $logStmt = $conn->prepare("
                    INSERT INTO mpesa_transactions (
                        order_id,
                        checkout_request_id,
                        phone_number,
                        amount,
                        description,
                        status,
                        created_at
                    ) VALUES (
                        :order_id,
                        :checkout_id,
                        :phone,
                        :amount,
                        :description,
                        'pending',
                        NOW()
                    )
                ");
                $logStmt->execute([
                    ':order_id' => $order_id,
                    ':checkout_id' => $response['checkout_id'],
                    ':phone' => $phone,
                    ':amount' => $total_amount,
                    ':description' => $description
                ]);
            } catch (Exception $e) {
                error_log('M-Pesa transaction logging failed: ' . $e->getMessage());
            }

            echo json_encode([
                'success' => true,
                'order_id' => $order_id,
                'order_number' => $order_number,
                'checkout_id' => $response['checkout_id'],
                'payment_initiated' => true,
                'message' => 'Payment request sent to your phone'
            ]);
            exit;
        }

        // M-Pesa initiation failed, fall back to cash
        $updateStmt = $conn->prepare("
            UPDATE orders 
            SET 
                payment_method = 'cash',
                status = 'pending',
                notes = CONCAT('M-Pesa failed: ', :reason),
                updated_at = NOW()
            WHERE id = :id
        ");
        $updateStmt->execute([
            ':reason' => $response['message'] ?? 'Payment initiation failed',
            ':id' => $order_id
        ]);

        echo json_encode([
            'success' => true,
            'order_id' => $order_id,
            'order_number' => $order_number,
            'payment_initiated' => false,
            'message' => 'Order placed. M-Pesa failed, please pay cash on delivery.'
        ]);
        exit;

    } catch (Exception $e) {
        // M-Pesa service error, fall back to cash
        error_log('M-Pesa service error: ' . $e->getMessage());
        
        if (isset($conn)) {
            $updateStmt = $conn->prepare("
                UPDATE orders 
                SET 
                    payment_method = 'cash',
                    status = 'pending',
                    notes = 'M-Pesa service error',
                    updated_at = NOW()
                WHERE id = :id
            ");
            $updateStmt->execute([':id' => $order_id]);
        }

        echo json_encode([
            'success' => true,
            'order_id' => $order_id,
            'order_number' => $order_number,
            'payment_initiated' => false,
            'message' => 'Order placed. Please pay cash on delivery.'
        ]);
        exit;
    }
}

/* -------------------------------------------------
   CASH ON DELIVERY
------------------------------------------------- */
// Update order status for cash payment
if (isset($conn)) {
    $updateStmt = $conn->prepare("
        UPDATE orders 
        SET 
            status = 'confirmed',
            updated_at = NOW()
        WHERE id = :id
    ");
    $updateStmt->execute([':id' => $order_id]);
}

echo json_encode([
    'success' => true,
    'order_id' => $order_id,
    'order_number' => $order_number,
    'payment_initiated' => false,
    'message' => 'Cash order placed successfully! Your order #' . $order_number . ' is being prepared.'
]);