<?php
// admin/combined_dashboard.php - Combined Outlet Dashboard

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

// Get date range filter
$period = $_GET['period'] ?? 'today';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

// Set date range based on period
$date_conditions = [];
$date_params = [];

if (!empty($start_date) && !empty($end_date)) {
    $date_conditions['sushi'] = "DATE(o.created_at) BETWEEN :start_date AND :end_date";
    $date_conditions['milano'] = "DATE(o.created_at) BETWEEN :start_date AND :end_date";
    $date_params[':start_date'] = $start_date;
    $date_params[':end_date'] = $end_date;
} else {
    switch ($period) {
        case 'today':
            $date_conditions['sushi'] = "DATE(o.created_at) = CURDATE()";
            $date_conditions['milano'] = "DATE(o.created_at) = CURDATE()";
            break;
        case 'yesterday':
            $date_conditions['sushi'] = "DATE(o.created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
            $date_conditions['milano'] = "DATE(o.created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
            break;
        case 'week':
            $date_conditions['sushi'] = "YEARWEEK(o.created_at, 1) = YEARWEEK(CURDATE(), 1)";
            $date_conditions['milano'] = "YEARWEEK(o.created_at, 1) = YEARWEEK(CURDATE(), 1)";
            break;
        case 'month':
            $date_conditions['sushi'] = "MONTH(o.created_at) = MONTH(CURDATE()) AND YEAR(o.created_at) = YEAR(CURDATE())";
            $date_conditions['milano'] = "MONTH(o.created_at) = MONTH(CURDATE()) AND YEAR(o.created_at) = YEAR(CURDATE())";
            break;
        default:
            $date_conditions['sushi'] = "1=1";
            $date_conditions['milano'] = "1=1";
    }
}

// Get statistics for both outlets
$outlets = ['sushi', 'milano'];
$outlet_stats = [];

foreach ($outlets as $outlet) {
    $date_condition = $date_conditions[$outlet] ?? "1=1";
    
    // Total orders
    $query = "SELECT COUNT(*) as total_orders FROM orders o WHERE o.outlet = :outlet AND $date_condition";
    $stmt = $conn->prepare($query);
    $params = [':outlet' => $outlet];
    if (!empty($date_params)) {
        $params = array_merge($params, $date_params);
    }
    $stmt->execute($params);
    $outlet_stats[$outlet]['total_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_orders'] ?? 0;
    
    // Total revenue
    $query = "SELECT SUM(o.total_amount) as total_revenue FROM orders o 
              WHERE o.outlet = :outlet AND $date_condition 
              AND (o.status = 'completed' OR o.status = 'paid')";
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $outlet_stats[$outlet]['total_revenue'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_revenue'] ?? 0;
    
    // Average order value
    $query = "SELECT AVG(o.total_amount) as avg_order_value FROM orders o 
              WHERE o.outlet = :outlet AND $date_condition 
              AND (o.status = 'completed' OR o.status = 'paid')";
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $outlet_stats[$outlet]['avg_order_value'] = $stmt->fetch(PDO::FETCH_ASSOC)['avg_order_value'] ?? 0;
    
    // Pending orders
    $query = "SELECT COUNT(*) as pending_orders FROM orders o 
              WHERE o.outlet = :outlet AND $date_condition 
              AND (o.status = 'pending' OR o.status = 'processing')";
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $outlet_stats[$outlet]['pending_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['pending_orders'] ?? 0;
    
    // Top items for this outlet
    $query = "SELECT oi.item_name, SUM(oi.quantity) as total_quantity, o.outlet
              FROM order_items oi
              JOIN orders o ON oi.order_id = o.id
              WHERE o.outlet = :outlet AND $date_condition
              GROUP BY oi.item_name, o.outlet
              ORDER BY total_quantity DESC 
              LIMIT 5";
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $outlet_stats[$outlet]['top_items'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Delivery locations for this outlet
    $query = "SELECT o.delivery_location, COUNT(*) as order_count, o.outlet
              FROM orders o 
              WHERE o.outlet = :outlet AND $date_condition
              GROUP BY o.delivery_location, o.outlet
              ORDER BY order_count DESC 
              LIMIT 5";
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $outlet_stats[$outlet]['top_locations'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get overall combined statistics
$combined_stats = [];

// Total combined orders
$query = "SELECT COUNT(*) as total_orders FROM orders o WHERE " . ($date_conditions['sushi'] ?? "1=1");
$stmt = $conn->prepare($query);
$stmt->execute($date_params);
$combined_stats['total_orders'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_orders'] ?? 0;

// Total combined revenue
$query = "SELECT SUM(o.total_amount) as total_revenue FROM orders o 
          WHERE (" . ($date_conditions['sushi'] ?? "1=1") . ")
          AND (o.status = 'completed' OR o.status = 'paid')";
$stmt = $conn->prepare($query);
$stmt->execute($date_params);
$combined_stats['total_revenue'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_revenue'] ?? 0;

// Recent orders from both outlets
$query = "SELECT o.* FROM orders o 
          WHERE (" . ($date_conditions['sushi'] ?? "1=1") . ")
          ORDER BY o.created_at DESC 
          LIMIT 15";
$stmt = $conn->prepare($query);
$stmt->execute($date_params);
$recent_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Order status breakdown
$query = "SELECT o.outlet, o.status, COUNT(*) as count 
          FROM orders o 
          WHERE (" . ($date_conditions['sushi'] ?? "1=1") . ")
          GROUP BY o.outlet, o.status";
$stmt = $conn->prepare($query);
$stmt->execute($date_params);
$status_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Organize status data by outlet
$status_breakdown = [];
foreach ($status_data as $row) {
    $outlet = $row['outlet'] ?? 'sushi';
    $status_breakdown[$outlet][$row['status']] = $row['count'];
}

// Get growth data (today vs yesterday)
$growth_data = [];
foreach ($outlets as $outlet) {
    // Today's orders
    $query = "SELECT COUNT(*) as count FROM orders o 
              WHERE o.outlet = :outlet AND DATE(o.created_at) = CURDATE()";
    $stmt = $conn->prepare($query);
    $stmt->execute([':outlet' => $outlet]);
    $today = $stmt->fetch(PDO::FETCH_ASSOC)['count'] ?? 0;
    
    // Yesterday's orders
    $query = "SELECT COUNT(*) as count FROM orders o 
              WHERE o.outlet = :outlet AND DATE(o.created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
    $stmt = $conn->prepare($query);
    $stmt->execute([':outlet' => $outlet]);
    $yesterday = $stmt->fetch(PDO::FETCH_ASSOC)['count'] ?? 0;
    
    // Calculate growth
    if ($yesterday > 0) {
        $growth = (($today - $yesterday) / $yesterday) * 100;
    } else {
        $growth = $today > 0 ? 100 : 0;
    }
    
    $growth_data[$outlet] = [
        'today' => $today,
        'yesterday' => $yesterday,
        'growth' => round($growth, 1)
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Combined Outlets Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        :root {
            --sushi-primary: #e53935;
            --sushi-primary-dark: #c62828;
            --milano-primary: #D35400;
            --milano-primary-dark: #A04000;
            --success: #2e7d32;
            --info: #17a2b8;
            --warning: #ffc107;
        }
        
        .outlet-card {
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 15px rgba(0,0,0,0.08);
            transition: transform 0.3s;
        }
        
        .outlet-card:hover {
            transform: translateY(-5px);
        }
        
        .outlet-card.sushi {
            border-left: 5px solid var(--sushi-primary);
            background: linear-gradient(135deg, rgba(229, 57, 53, 0.1), rgba(229, 57, 53, 0.05));
        }
        
        .outlet-card.milano {
            border-left: 5px solid var(--milano-primary);
            background: linear-gradient(135deg, rgba(211, 84, 0, 0.1), rgba(211, 84, 0, 0.05));
        }
        
        .combined-card {
            border-left: 5px solid var(--info);
            background: linear-gradient(135deg, rgba(23, 162, 184, 0.1), rgba(23, 162, 184, 0.05));
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 800;
            line-height: 1;
        }
        
        .stat-label {
            color: #666;
            font-size: 0.9rem;
            margin-top: 5px;
        }
        
        .growth-badge {
            font-size: 0.8rem;
            padding: 3px 8px;
            border-radius: 15px;
        }
        
        .growth-up {
            background: #d4edda;
            color: #155724;
        }
        
        .growth-down {
            background: #f8d7da;
            color: #721c24;
        }
        
        .outlet-badge {
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            display: inline-block;
            margin-bottom: 10px;
        }
        
        .badge-sushi {
            background: linear-gradient(135deg, var(--sushi-primary), var(--sushi-primary-dark));
            color: white;
        }
        
        .badge-milano {
            background: linear-gradient(135deg, var(--milano-primary), var(--milano-primary-dark));
            color: white;
        }
        
        .status-badge {
            padding: 5px 10px;
            border-radius: 5px;
            font-size: 0.8rem;
        }
        
        .badge-pending { background: #fff3cd; color: #856404; }
        .badge-processing { background: #d1ecf1; color: #0c5460; }
        .badge-completed { background: #d4edda; color: #155724; }
        .badge-paid { background: #d1e7dd; color: #0f5132; }
        .badge-cancelled { background: #f8d7da; color: #721c24; }
        
        .chart-container {
            height: 300px;
            position: relative;
        }
        
        .period-selector .btn {
            border-radius: 5px;
            padding: 5px 15px;
            font-size: 0.9rem;
        }
        
        .period-selector .btn.active {
            background: var(--info);
            color: white;
            border-color: var(--info);
        }
        
        .comparison-bar {
            height: 10px;
            border-radius: 5px;
            margin: 5px 0;
            position: relative;
        }
        
        .comparison-bar.sushi {
            background: linear-gradient(to right, var(--sushi-primary), var(--sushi-primary-dark));
        }
        
        .comparison-bar.milano {
            background: linear-gradient(to right, var(--milano-primary), var(--milano-primary-dark));
        }
        
        .comparison-label {
            font-size: 0.8rem;
            color: #666;
            display: flex;
            justify-content: space-between;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-12">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2><i class="fas fa-store-alt"></i> Combined Outlets Dashboard</h2>
                        <p class="text-muted mb-0">Overview of both SUSHI-BAR and MILANO PIZZA-PASTA</p>
                    </div>
                    <div>
                        <a href="index.php?outlet=sushi" class="btn btn-outline-danger">
                            <i class="fas fa-fish"></i> Sushi Dashboard
                        </a>
                        <a href="index.php?outlet=milano" class="btn btn-outline-warning">
                            <i class="fas fa-pizza-slice"></i> Milano Dashboard
                        </a>
                    </div>
                </div>
                
                <!-- Period Selector -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="period-selector">
                                <a href="?period=today" class="btn btn-sm <?php echo $period === 'today' ? 'active' : 'btn-outline-secondary'; ?>">
                                    Today
                                </a>
                                <a href="?period=yesterday" class="btn btn-sm <?php echo $period === 'yesterday' ? 'active' : 'btn-outline-secondary'; ?>">
                                    Yesterday
                                </a>
                                <a href="?period=week" class="btn btn-sm <?php echo $period === 'week' ? 'active' : 'btn-outline-secondary'; ?>">
                                    This Week
                                </a>
                                <a href="?period=month" class="btn btn-sm <?php echo $period === 'month' ? 'active' : 'btn-outline-secondary'; ?>">
                                    This Month
                                </a>
                                <a href="?period=all" class="btn btn-sm <?php echo ($period === 'all' || empty($period)) ? 'active' : 'btn-outline-secondary'; ?>">
                                    All Time
                                </a>
                            </div>
                            
                            <div class="d-flex">
                                <input type="date" 
                                       class="form-control form-control-sm me-2" 
                                       id="startDate" 
                                       value="<?php echo $start_date; ?>"
                                       style="width: 150px;">
                                <span class="align-self-center me-2">to</span>
                                <input type="date" 
                                       class="form-control form-control-sm me-2" 
                                       id="endDate" 
                                       value="<?php echo $end_date; ?>"
                                       style="width: 150px;">
                                <button class="btn btn-primary btn-sm" onclick="applyCustomDate()">
                                    <i class="fas fa-filter"></i> Apply
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Combined Overview -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="outlet-card combined-card">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <div class="text-muted small">TOTAL ORDERS</div>
                                    <div class="stat-number"><?php echo number_format($combined_stats['total_orders']); ?></div>
                                    <div class="stat-label">Across Both Outlets</div>
                                </div>
                                <div>
                                    <i class="fas fa-shopping-cart fa-2x text-info"></i>
                                </div>
                            </div>
                            <div class="mt-3">
                                <div class="comparison-label">
                                    <span>SUSHI-BAR</span>
                                    <span><?php echo number_format($outlet_stats['sushi']['total_orders']); ?></span>
                                </div>
                                <div class="comparison-bar sushi" style="width: <?php echo $combined_stats['total_orders'] > 0 ? ($outlet_stats['sushi']['total_orders'] / $combined_stats['total_orders'] * 100) : 0; ?>%"></div>
                                
                                <div class="comparison-label mt-2">
                                    <span>MILANO</span>
                                    <span><?php echo number_format($outlet_stats['milano']['total_orders']); ?></span>
                                </div>
                                <div class="comparison-bar milano" style="width: <?php echo $combined_stats['total_orders'] > 0 ? ($outlet_stats['milano']['total_orders'] / $combined_stats['total_orders'] * 100) : 0; ?>%"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="outlet-card combined-card">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <div class="text-muted small">TOTAL REVENUE</div>
                                    <div class="stat-number">KES <?php echo number_format($combined_stats['total_revenue'], 2); ?></div>
                                    <div class="stat-label">Across Both Outlets</div>
                                </div>
                                <div>
                                    <i class="fas fa-money-bill-wave fa-2x text-info"></i>
                                </div>
                            </div>
                            <div class="mt-3">
                                <div class="comparison-label">
                                    <span>SUSHI-BAR</span>
                                    <span>KES <?php echo number_format($outlet_stats['sushi']['total_revenue'], 2); ?></span>
                                </div>
                                <div class="comparison-bar sushi" style="width: <?php echo $combined_stats['total_revenue'] > 0 ? ($outlet_stats['sushi']['total_revenue'] / $combined_stats['total_revenue'] * 100) : 0; ?>%"></div>
                                
                                <div class="comparison-label mt-2">
                                    <span>MILANO</span>
                                    <span>KES <?php echo number_format($outlet_stats['milano']['total_revenue'], 2); ?></span>
                                </div>
                                <div class="comparison-bar milano" style="width: <?php echo $combined_stats['total_revenue'] > 0 ? ($outlet_stats['milano']['total_revenue'] / $combined_stats['total_revenue'] * 100) : 0; ?>%"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="outlet-card combined-card">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <div class="text-muted small">PENDING ORDERS</div>
                                    <div class="stat-number"><?php echo number_format($outlet_stats['sushi']['pending_orders'] + $outlet_stats['milano']['pending_orders']); ?></div>
                                    <div class="stat-label">Awaiting Processing</div>
                                </div>
                                <div>
                                    <i class="fas fa-clock fa-2x text-warning"></i>
                                </div>
                            </div>
                            <div class="mt-3">
                                <div class="comparison-label">
                                    <span>SUSHI-BAR Pending</span>
                                    <span><?php echo number_format($outlet_stats['sushi']['pending_orders']); ?></span>
                                </div>
                                <div class="comparison-bar sushi" style="width: <?php echo ($outlet_stats['sushi']['pending_orders'] + $outlet_stats['milano']['pending_orders']) > 0 ? ($outlet_stats['sushi']['pending_orders'] / ($outlet_stats['sushi']['pending_orders'] + $outlet_stats['milano']['pending_orders']) * 100) : 0; ?>%"></div>
                                
                                <div class="comparison-label mt-2">
                                    <span>MILANO Pending</span>
                                    <span><?php echo number_format($outlet_stats['milano']['pending_orders']); ?></span>
                                </div>
                                <div class="comparison-bar milano" style="width: <?php echo ($outlet_stats['sushi']['pending_orders'] + $outlet_stats['milano']['pending_orders']) > 0 ? ($outlet_stats['milano']['pending_orders'] / ($outlet_stats['sushi']['pending_orders'] + $outlet_stats['milano']['pending_orders']) * 100) : 0; ?>%"></div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Outlet Comparison -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-line"></i> Outlet Comparison</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="outlet-card sushi">
                                            <span class="outlet-badge badge-sushi">
                                                <i class="fas fa-fish"></i> SUSHI-BAR
                                            </span>
                                            <div class="d-flex justify-content-between mt-3">
                                                <div>
                                                    <div class="text-muted small">ORDERS TODAY</div>
                                                    <div class="stat-number"><?php echo number_format($growth_data['sushi']['today']); ?></div>
                                                    <div class="d-flex align-items-center mt-1">
                                                        <?php if ($growth_data['sushi']['growth'] >= 0): ?>
                                                        <span class="growth-badge growth-up me-2">
                                                            <i class="fas fa-arrow-up"></i> <?php echo $growth_data['sushi']['growth']; ?>%
                                                        </span>
                                                        <?php else: ?>
                                                        <span class="growth-badge growth-down me-2">
                                                            <i class="fas fa-arrow-down"></i> <?php echo abs($growth_data['sushi']['growth']); ?>%
                                                        </span>
                                                        <?php endif; ?>
                                                        <span class="text-muted small">vs yesterday</span>
                                                    </div>
                                                </div>
                                                <div class="text-end">
                                                    <div class="text-muted small">AVG ORDER</div>
                                                    <div class="h4">KES <?php echo number_format($outlet_stats['sushi']['avg_order_value'], 2); ?></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="outlet-card milano">
                                            <span class="outlet-badge badge-milano">
                                                <i class="fas fa-pizza-slice"></i> MILANO
                                            </span>
                                            <div class="d-flex justify-content-between mt-3">
                                                <div>
                                                    <div class="text-muted small">ORDERS TODAY</div>
                                                    <div class="stat-number"><?php echo number_format($growth_data['milano']['today']); ?></div>
                                                    <div class="d-flex align-items-center mt-1">
                                                        <?php if ($growth_data['milano']['growth'] >= 0): ?>
                                                        <span class="growth-badge growth-up me-2">
                                                            <i class="fas fa-arrow-up"></i> <?php echo $growth_data['milano']['growth']; ?>%
                                                        </span>
                                                        <?php else: ?>
                                                        <span class="growth-badge growth-down me-2">
                                                            <i class="fas fa-arrow-down"></i> <?php echo abs($growth_data['milano']['growth']); ?>%
                                                        </span>
                                                        <?php endif; ?>
                                                        <span class="text-muted small">vs yesterday</span>
                                                    </div>
                                                </div>
                                                <div class="text-end">
                                                    <div class="text-muted small">AVG ORDER</div>
                                                    <div class="h4">KES <?php echo number_format($outlet_stats['milano']['avg_order_value'], 2); ?></div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-pie"></i> Order Status Breakdown</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <div class="outlet-badge badge-sushi mb-2">SUSHI-BAR Status</div>
                                            <?php if (!empty($status_breakdown['sushi'])): ?>
                                            <?php foreach ($status_breakdown['sushi'] as $status => $count): ?>
                                            <div class="d-flex justify-content-between align-items-center mb-2">
                                                <span class="badge status-badge badge-<?php echo $status; ?>">
                                                    <?php echo ucfirst($status); ?>
                                                </span>
                                                <span class="fw-bold"><?php echo $count; ?></span>
                                            </div>
                                            <?php endforeach; ?>
                                            <?php else: ?>
                                            <p class="text-muted small">No orders for this period</p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <div class="outlet-badge badge-milano mb-2">MILANO Status</div>
                                            <?php if (!empty($status_breakdown['milano'])): ?>
                                            <?php foreach ($status_breakdown['milano'] as $status => $count): ?>
                                            <div class="d-flex justify-content-between align-items-center mb-2">
                                                <span class="badge status-badge badge-<?php echo $status; ?>">
                                                    <?php echo ucfirst($status); ?>
                                                </span>
                                                <span class="fw-bold"><?php echo $count; ?></span>
                                            </div>
                                            <?php endforeach; ?>
                                            <?php else: ?>
                                            <p class="text-muted small">No orders for this period</p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Orders -->
                <div class="row mb-4">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0"><i class="fas fa-history"></i> Recent Orders (Both Outlets)</h5>
                                <a href="orders.php" class="btn btn-primary btn-sm">View All Orders</a>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table table-hover mb-0">
                                        <thead>
                                            <tr>
                                                <th>Outlet</th>
                                                <th>Order #</th>
                                                <th>Customer</th>
                                                <th>Amount</th>
                                                <th>Status</th>
                                                <th>Date</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($recent_orders as $order): ?>
                                            <tr>
                                                <td>
                                                    <?php if ($order['outlet'] === 'milano'): ?>
                                                    <span class="badge badge-milano">
                                                        <i class="fas fa-pizza-slice"></i> MILANO
                                                    </span>
                                                    <?php else: ?>
                                                    <span class="badge badge-sushi">
                                                        <i class="fas fa-fish"></i> SUSHI
                                                    </span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($order['order_number']); ?></strong>
                                                </td>
                                                <td>
                                                    <div><?php echo htmlspecialchars($order['customer_name']); ?></div>
                                                    <small class="text-muted"><?php echo htmlspecialchars($order['customer_phone']); ?></small>
                                                </td>
                                                <td>KES <?php echo number_format($order['total_amount'], 2); ?></td>
                                                <td>
                                                    <?php
                                                    $status_class = 'badge-pending';
                                                    switch($order['status']) {
                                                        case 'pending': $status_class = 'badge-pending'; break;
                                                        case 'processing': $status_class = 'badge-processing'; break;
                                                        case 'completed': $status_class = 'badge-completed'; break;
                                                        case 'paid': $status_class = 'badge-paid'; break;
                                                        case 'cancelled': $status_class = 'badge-cancelled'; break;
                                                    }
                                                    ?>
                                                    <span class="badge status-badge <?php echo $status_class; ?>">
                                                        <?php echo ucfirst($order['status']); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo date('M j, g:i A', strtotime($order['created_at'])); ?></td>
                                                <td>
                                                    <a href="order_details.php?id=<?php echo $order['id']; ?>&outlet=<?php echo $order['outlet']; ?>" 
                                                       class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <?php if (empty($recent_orders)): ?>
                                            <tr>
                                                <td colspan="7" class="text-center py-4">
                                                    <i class="fas fa-shopping-cart fa-2x text-muted mb-3"></i>
                                                    <p class="text-muted">No recent orders found</p>
                                                </td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Top Items & Locations -->
                    <div class="col-md-4">
                        <!-- Top Items -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-star"></i> Top Items</h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <div class="outlet-badge badge-sushi mb-2">SUSHI-BAR Top Items</div>
                                    <?php if (!empty($outlet_stats['sushi']['top_items'])): ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($outlet_stats['sushi']['top_items'] as $item): ?>
                                        <li class="list-group-item d-flex justify-content-between align-items-center py-2">
                                            <span class="small"><?php echo htmlspecialchars($item['item_name']); ?></span>
                                            <span class="badge bg-danger rounded-pill"><?php echo $item['total_quantity']; ?></span>
                                        </li>
                                        <?php endforeach; ?>
                                    </ul>
                                    <?php else: ?>
                                    <p class="text-muted small">No items sold</p>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="mt-3">
                                    <div class="outlet-badge badge-milano mb-2">MILANO Top Items</div>
                                    <?php if (!empty($outlet_stats['milano']['top_items'])): ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($outlet_stats['milano']['top_items'] as $item): ?>
                                        <li class="list-group-item d-flex justify-content-between align-items-center py-2">
                                            <span class="small"><?php echo htmlspecialchars($item['item_name']); ?></span>
                                            <span class="badge bg-warning rounded-pill"><?php echo $item['total_quantity']; ?></span>
                                        </li>
                                        <?php endforeach; ?>
                                    </ul>
                                    <?php else: ?>
                                    <p class="text-muted small">No items sold</p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Top Locations -->
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-map-marker-alt"></i> Top Delivery Locations</h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <div class="outlet-badge badge-sushi mb-2">SUSHI-BAR Locations</div>
                                    <?php if (!empty($outlet_stats['sushi']['top_locations'])): ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($outlet_stats['sushi']['top_locations'] as $location): ?>
                                        <li class="list-group-item d-flex justify-content-between align-items-center py-2">
                                            <span class="small"><?php echo htmlspecialchars($location['delivery_location']); ?></span>
                                            <span class="badge bg-success rounded-pill"><?php echo $location['order_count']; ?></span>
                                        </li>
                                        <?php endforeach; ?>
                                    </ul>
                                    <?php else: ?>
                                    <p class="text-muted small">No delivery data</p>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="mt-3">
                                    <div class="outlet-badge badge-milano mb-2">MILANO Locations</div>
                                    <?php if (!empty($outlet_stats['milano']['top_locations'])): ?>
                                    <ul class="list-group list-group-flush">
                                        <?php foreach ($outlet_stats['milano']['top_locations'] as $location): ?>
                                        <li class="list-group-item d-flex justify-content-between align-items-center py-2">
                                            <span class="small"><?php echo htmlspecialchars($location['delivery_location']); ?></span>
                                            <span class="badge bg-success rounded-pill"><?php echo $location['order_count']; ?></span>
                                        </li>
                                        <?php endforeach; ?>
                                    </ul>
                                    <?php else: ?>
                                    <p class="text-muted small">No delivery data</p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function applyCustomDate() {
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            
            if (startDate && endDate) {
                window.location.href = `?start_date=${startDate}&end_date=${endDate}`;
            } else {
                alert('Please select both start and end dates');
            }
        }
        
        // Auto-refresh dashboard every 60 seconds
        setInterval(function() {
            const currentUrl = window.location.href;
            if (!currentUrl.includes('action=')) {
                window.location.reload();
            }
        }, 60000);
        
        // Set today as default end date if not set
        document.addEventListener('DOMContentLoaded', function() {
            const endDateInput = document.getElementById('endDate');
            const startDateInput = document.getElementById('startDate');
            
            if (endDateInput && !endDateInput.value) {
                const today = new Date().toISOString().split('T')[0];
                endDateInput.value = today;
                
                // Set start date to 7 days ago by default
                const weekAgo = new Date();
                weekAgo.setDate(weekAgo.getDate() - 7);
                if (!startDateInput.value) {
                    startDateInput.value = weekAgo.toISOString().split('T')[0];
                }
            }
        });
    </script>
</body>
</html>