<?php
// admin/customer_feedback.php - Admin Feedback Management

// Simple authentication (you should enhance this for production)
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

$feedback_file = 'customer_feedback.php';
$all_feedback = [];

// Read feedback data
if (file_exists($feedback_file)) {
    $content = file_get_contents($feedback_file);
    $json_start = strpos($content, '[');
    if ($json_start !== false) {
        $json_content = substr($content, $json_start);
        $all_feedback = json_decode($json_content, true);
        if (!is_array($all_feedback)) {
            $all_feedback = [];
        }
    }
}

// Handle actions (update status, delete)
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $feedback_id = $_POST['feedback_id'] ?? '';
    
    if ($action == 'update_status' && $feedback_id) {
        $new_status = $_POST['status'] ?? '';
        $admin_notes = $_POST['admin_notes'] ?? '';
        
        // Update feedback status
        foreach ($all_feedback as &$feedback) {
            if ($feedback['id'] == $feedback_id) {
                $feedback['status'] = $new_status;
                $feedback['admin_notes'] = $admin_notes;
                if ($new_status == 'resolved') {
                    $feedback['resolved_at'] = date('Y-m-d H:i:s');
                }
                break;
            }
        }
        
        saveFeedbackData($all_feedback);
        
    } elseif ($action == 'delete' && $feedback_id) {
        // Remove feedback
        $all_feedback = array_filter($all_feedback, function($feedback) use ($feedback_id) {
            return $feedback['id'] != $feedback_id;
        });
        
        saveFeedbackData($all_feedback);
        $all_feedback = array_values($all_feedback); // Reset array keys
    }
}

function saveFeedbackData($data) {
    global $feedback_file;
    $file_content = '<?php
/**
 * SUSHI-BAR Customer Feedback
 * This file stores customer feedback submissions
 * Access this page through admin panel with proper authentication
 */
header("Content-Type: application/json");
?>';
    $file_content .= "\n\n" . json_encode($data, JSON_PRETTY_PRINT);
    file_put_contents($feedback_file, $file_content);
}

// Get statistics
$stats = [
    'total' => count($all_feedback),
    'pending' => 0,
    'reviewed' => 0,
    'resolved' => 0,
    'avg_rating' => 0
];

$total_rating = 0;
foreach ($all_feedback as $feedback) {
    $status = $feedback['status'] ?? 'pending';
    $stats[$status]++;
    $total_rating += $feedback['rating'];
}

$stats['avg_rating'] = $stats['total'] > 0 ? round($total_rating / $stats['total'], 1) : 0;

// Sort feedback by date (newest first)
usort($all_feedback, function($a, $b) {
    return strtotime($b['submitted_at'] ?? '') - strtotime($a['submitted_at'] ?? '');
});
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Feedback - SUSHI-BAR Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        :root {
            --primary: #e53935;
            --primary-dark: #c62828;
            --secondary: #25D366;
            --light: #f5f5f5;
            --dark: #333;
            --gray: #777;
            --light-gray: #f9f9f9;
            --border: #ddd;
            --shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        body {
            background-color: #f8f8f8;
            color: var(--dark);
            line-height: 1.6;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        /* Header */
        .admin-header {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            padding: 20px 0;
            box-shadow: var(--shadow);
            margin-bottom: 30px;
        }
        
        .admin-header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .admin-logo {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .admin-logo img {
            height: 40px;
            width: auto;
        }
        
        .admin-logo h1 {
            font-size: 1.5rem;
            font-weight: 700;
        }
        
        .admin-nav {
            display: flex;
            gap: 20px;
            align-items: center;
        }
        
        .admin-nav a {
            color: white;
            text-decoration: none;
            padding: 8px 15px;
            border-radius: 4px;
            transition: background 0.3s;
        }
        
        .admin-nav a:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .logout-btn {
            background: rgba(255, 255, 255, 0.2);
            color: white;
            padding: 8px 15px;
            border-radius: 4px;
            text-decoration: none;
            transition: background 0.3s;
        }
        
        .logout-btn:hover {
            background: rgba(255, 255, 255, 0.3);
        }
        
        /* Stats Cards */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: var(--shadow);
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: white;
        }
        
        .stat-icon.pending { background: #ff9800; }
        .stat-icon.reviewed { background: #2196f3; }
        .stat-icon.resolved { background: #4caf50; }
        .stat-icon.total { background: #9c27b0; }
        .stat-icon.rating { background: #ff9800; }
        
        .stat-info h3 {
            font-size: 1.8rem;
            margin-bottom: 5px;
        }
        
        .stat-info p {
            color: var(--gray);
            font-size: 0.9rem;
        }
        
        /* Feedback Table */
        .feedback-table-container {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: var(--shadow);
            margin-bottom: 30px;
            overflow-x: auto;
        }
        
        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            background: var(--light-gray);
            padding: 15px;
            text-align: left;
            font-weight: 600;
            border-bottom: 2px solid var(--border);
        }
        
        td {
            padding: 15px;
            border-bottom: 1px solid var(--border);
            vertical-align: top;
        }
        
        tr:hover {
            background: var(--light-gray);
        }
        
        /* Status badges */
        .status-badge {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .status-pending {
            background: #ffecb3;
            color: #ff8f00;
        }
        
        .status-reviewed {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .status-resolved {
            background: #e8f5e9;
            color: #388e3c;
        }
        
        /* Rating stars */
        .rating-stars {
            color: #ff9800;
            font-size: 1.1rem;
        }
        
        /* Feedback type icons */
        .feedback-type-icon {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.9rem;
        }
        
        .type-compliment {
            background: #e8f5e9;
            color: #388e3c;
        }
        
        .type-suggestion {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .type-complaint {
            background: #ffebee;
            color: #d32f2f;
        }
        
        .type-other {
            background: #f3e5f5;
            color: #7b1fa2;
        }
        
        /* Action buttons */
        .action-buttons {
            display: flex;
            gap: 5px;
        }
        
        .action-btn {
            padding: 5px 10px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: all 0.3s;
        }
        
        .edit-btn {
            background: #2196f3;
            color: white;
        }
        
        .edit-btn:hover {
            background: #1976d2;
        }
        
        .delete-btn {
            background: #f44336;
            color: white;
        }
        
        .delete-btn:hover {
            background: #d32f2f;
        }
        
        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            width: 90%;
            max-width: 600px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow);
            animation: fadeIn 0.3s ease-out;
        }
        
        .modal-header {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            padding: 20px;
            border-radius: 10px 10px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .close-modal {
            background: none;
            border: none;
            color: white;
            font-size: 1.5rem;
            cursor: pointer;
        }
        
        .modal-body {
            padding: 20px;
        }
        
        .feedback-detail {
            margin-bottom: 20px;
        }
        
        .detail-item {
            margin-bottom: 15px;
        }
        
        .detail-label {
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .detail-value {
            padding: 10px;
            background: var(--light-gray);
            border-radius: 5px;
            border-left: 3px solid var(--primary);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
        }
        
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid var(--border);
            border-radius: 5px;
        }
        
        textarea.form-control {
            min-height: 100px;
            resize: vertical;
        }
        
        .modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
            padding: 20px;
            border-top: 1px solid var(--border);
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 600;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-primary:hover {
            background: var(--primary-dark);
        }
        
        .btn-secondary {
            background: var(--light-gray);
            color: var(--dark);
        }
        
        .btn-secondary:hover {
            background: #e0e0e0;
        }
        
        /* Empty state */
        .empty-state {
            text-align: center;
            padding: 50px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.3;
        }
        
        /* Animations */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .admin-header-content {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .admin-nav {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stats-container {
                grid-template-columns: repeat(2, 1fr);
            }
            
            th, td {
                padding: 10px;
                font-size: 0.9rem;
            }
        }
    </style>
</head>
<body>
    <!-- Admin Header -->
    <header class="admin-header">
        <div class="container admin-header-content">
            <div class="admin-logo">
                <img src="https://aromacafe.co.ke/blaqpremium/images/sushilogo.png" 
                     alt="SUSHI-BAR Logo">
                <h1>Customer Feedback Management</h1>
            </div>
            <div class="admin-nav">
                <a href="dashboard.php">Dashboard</a>
                <a href="orders.php">Orders</a>
                <a href="menu.php">Menu</a>
                <a href="customer_feedback.php" style="background: rgba(255,255,255,0.2);">Feedback</a>
                <a href="settings.php">Settings</a>
                <a href="logout.php" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </header>

    <div class="container">
        <!-- Stats Cards -->
        <div class="stats-container">
            <div class="stat-card">
                <div class="stat-icon total">
                    <i class="fas fa-comments"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['total']; ?></h3>
                    <p>Total Feedback</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon pending">
                    <i class="fas fa-clock"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['pending']; ?></h3>
                    <p>Pending Review</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon resolved">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['resolved']; ?></h3>
                    <p>Resolved</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon rating">
                    <i class="fas fa-star"></i>
                </div>
                <div class="stat-info">
                    <h3><?php echo $stats['avg_rating']; ?>/5</h3>
                    <p>Average Rating</p>
                </div>
            </div>
        </div>

        <!-- Feedback Table -->
        <div class="feedback-table-container">
            <div class="table-header">
                <h2>Customer Feedback</h2>
            </div>
            
            <?php if (empty($all_feedback)): ?>
                <div class="empty-state">
                    <i class="fas fa-comment-slash"></i>
                    <h3>No feedback submissions yet</h3>
                    <p>Customer feedback will appear here once submitted</p>
                </div>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Customer</th>
                            <th>Type</th>
                            <th>Rating</th>
                            <th>Feedback</th>
                            <th>Status</th>
                            <th>Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($all_feedback as $feedback): ?>
                            <tr>
                                <td><?php echo substr($feedback['id'], 0, 8) . '...'; ?></td>
                                <td>
                                    <strong><?php echo htmlspecialchars($feedback['name']); ?></strong><br>
                                    <?php if (!empty($feedback['email'])): ?>
                                        <small><?php echo htmlspecialchars($feedback['email']); ?></small><br>
                                    <?php endif; ?>
                                    <?php if (!empty($feedback['phone'])): ?>
                                        <small><?php echo htmlspecialchars($feedback['phone']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="feedback-type-icon type-<?php echo strtolower($feedback['feedback_type']); ?>">
                                        <?php if ($feedback['feedback_type'] == 'Compliment'): ?>
                                            <i class="fas fa-heart"></i>
                                        <?php elseif ($feedback['feedback_type'] == 'Suggestion'): ?>
                                            <i class="fas fa-lightbulb"></i>
                                        <?php elseif ($feedback['feedback_type'] == 'Complaint'): ?>
                                            <i class="fas fa-exclamation-triangle"></i>
                                        <?php else: ?>
                                            <i class="fas fa-comment"></i>
                                        <?php endif; ?>
                                        <?php echo $feedback['feedback_type']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="rating-stars" title="<?php echo $feedback['rating']; ?>/5">
                                        <?php echo str_repeat('★', $feedback['rating']); ?>
                                        <?php echo str_repeat('☆', 5 - $feedback['rating']); ?>
                                    </div>
                                </td>
                                <td>
                                    <?php 
                                    $short_feedback = strlen($feedback['feedback']) > 100 
                                        ? substr($feedback['feedback'], 0, 100) . '...' 
                                        : $feedback['feedback'];
                                    echo htmlspecialchars($short_feedback);
                                    ?>
                                    <?php if (!empty($feedback['order_number'])): ?>
                                        <br><small>Order: <?php echo htmlspecialchars($feedback['order_number']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo $feedback['status']; ?>">
                                        <?php echo ucfirst($feedback['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo date('M d, Y', strtotime($feedback['submitted_at'])); ?><br>
                                    <small><?php echo date('H:i', strtotime($feedback['submitted_at'])); ?></small>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="action-btn edit-btn" 
                                                onclick="openFeedbackModal('<?php echo $feedback['id']; ?>')">
                                            <i class="fas fa-edit"></i> View/Edit
                                        </button>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="feedback_id" value="<?php echo $feedback['id']; ?>">
                                            <input type="hidden" name="action" value="delete">
                                            <button type="submit" class="action-btn delete-btn" 
                                                    onclick="return confirm('Are you sure you want to delete this feedback?')">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <!-- Feedback Modal -->
    <div class="modal" id="feedbackModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Feedback Details</h3>
                <button class="close-modal" onclick="closeFeedbackModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <div id="feedbackDetails"></div>
                <form id="feedbackForm" method="POST">
                    <input type="hidden" name="feedback_id" id="modalFeedbackId">
                    <input type="hidden" name="action" value="update_status">
                    
                    <div class="form-group">
                        <label class="form-label">Update Status</label>
                        <select name="status" id="statusSelect" class="form-control" required>
                            <option value="pending">Pending</option>
                            <option value="reviewed">Reviewed</option>
                            <option value="resolved">Resolved</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Admin Notes</label>
                        <textarea name="admin_notes" id="adminNotes" class="form-control" 
                                  placeholder="Add internal notes about this feedback..."></textarea>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" onclick="closeFeedbackModal()">
                            Cancel
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Feedback data from PHP
        const feedbackData = <?php echo json_encode($all_feedback); ?>;
        
        function openFeedbackModal(feedbackId) {
            const feedback = feedbackData.find(f => f.id === feedbackId);
            if (!feedback) return;
            
            // Update modal title and ID
            document.getElementById('modalFeedbackId').value = feedbackId;
            document.getElementById('statusSelect').value = feedback.status || 'pending';
            document.getElementById('adminNotes').value = feedback.admin_notes || '';
            
            // Populate feedback details
            const detailsHtml = `
                <div class="feedback-detail">
                    <div class="detail-item">
                        <div class="detail-label">Customer Information</div>
                        <div class="detail-value">
                            <strong>${feedback.name}</strong><br>
                            ${feedback.email ? `Email: ${feedback.email}<br>` : ''}
                            ${feedback.phone ? `Phone: ${feedback.phone}<br>` : ''}
                            ${feedback.order_number ? `Order #: ${feedback.order_number}` : ''}
                        </div>
                    </div>
                    
                    <div class="detail-item">
                        <div class="detail-label">Feedback Type & Rating</div>
                        <div class="detail-value">
                            <span style="display: inline-flex; align-items: center; gap: 10px;">
                                <span style="padding: 5px 10px; border-radius: 4px; background: ${getTypeColor(feedback.feedback_type)};">
                                    ${feedback.feedback_type}
                                </span>
                                <span style="color: #ff9800; font-size: 1.2rem;">
                                    ${'★'.repeat(feedback.rating)}${'☆'.repeat(5 - feedback.rating)}
                                    (${feedback.rating}/5)
                                </span>
                            </span>
                        </div>
                    </div>
                    
                    <div class="detail-item">
                        <div class="detail-label">Feedback</div>
                        <div class="detail-value" style="white-space: pre-wrap;">${feedback.feedback}</div>
                    </div>
                    
                    <div class="detail-item">
                        <div class="detail-label">Submission Details</div>
                        <div class="detail-value">
                            Submitted: ${new Date(feedback.submitted_at).toLocaleString()}<br>
                            ${feedback.contact_consent == '1' ? '✓ Customer gave contact consent' : 'No contact consent given'}
                        </div>
                    </div>
                    
                    ${feedback.resolved_at ? `
                    <div class="detail-item">
                        <div class="detail-label">Resolution</div>
                        <div class="detail-value">
                            Resolved on: ${new Date(feedback.resolved_at).toLocaleString()}
                        </div>
                    </div>
                    ` : ''}
                </div>
            `;
            
            document.getElementById('feedbackDetails').innerHTML = detailsHtml;
            document.getElementById('feedbackModal').classList.add('active');
        }
        
        function closeFeedbackModal() {
            document.getElementById('feedbackModal').classList.remove('active');
        }
        
        function getTypeColor(type) {
            switch(type) {
                case 'Compliment': return '#e8f5e9';
                case 'Suggestion': return '#e3f2fd';
                case 'Complaint': return '#ffebee';
                default: return '#f3e5f5';
            }
        }
        
        // Close modal on outside click
        document.getElementById('feedbackModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeFeedbackModal();
            }
        });
        
        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeFeedbackModal();
            }
        });
    </script>
</body>
</html>