<?php
// admin/menu_save.php
session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error_message'] = "Invalid request method";
    header('Location: menu.php');
    exit;
}

// Get form data
$id = $_POST['id'] ?? null;
$name = trim($_POST['name'] ?? '');
$description = trim($_POST['description'] ?? '');
$category = trim($_POST['category'] ?? 'pizza');
$price = floatval($_POST['price'] ?? 0);
$sizes = $_POST['sizes'] ?? [];
$pizza_category_id = $_POST['pizza_category_id'] ?? null;
$other_category = trim($_POST['other_category'] ?? '');
$is_active = isset($_POST['is_active']) ? 1 : 0;
$outlet = $_POST['outlet'] ?? 'milano';
$current_image = $_POST['current_image'] ?? '';
$item_type = $_POST['item_type'] ?? 'regular'; // 'pizza' or 'regular'

// Validation
if (empty($name)) {
    $_SESSION['error_message'] = "Name is required";
    header("Location: menu.php?outlet=$outlet");
    exit;
}

if ($price <= 0) {
    $_SESSION['error_message'] = "Price must be greater than 0";
    header("Location: menu.php?outlet=$outlet");
    exit;
}

// Handle category for regular items
if ($item_type === 'regular') {
    // Use "other_category" if category is set to "other"
    if ($category === 'other' && !empty($other_category)) {
        $category = trim($other_category);
    } elseif ($category === 'other' && empty($other_category)) {
        $_SESSION['error_message'] = "Please enter a category name";
        header("Location: menu.php?outlet=$outlet");
        exit;
    }
}

// Process sizes only for pizza items
$sizesJson = null;
if ($item_type === 'pizza') {
    if (!empty($sizes) && is_array($sizes)) {
        $filteredSizes = [];
        foreach ($sizes as $size => $sizePrice) {
            $sizePrice = floatval($sizePrice);
            if ($sizePrice > 0) {
                $filteredSizes[$size] = $sizePrice;
            }
        }
        
        // If no sizes are set but price is set, use price as small size
        if (empty($filteredSizes) && $price > 0) {
            $filteredSizes['small'] = $price;
        }
        
        if (!empty($filteredSizes)) {
            $sizesJson = json_encode($filteredSizes);
        }
    } else if ($price > 0) {
        // Default to small size with base price
        $sizesJson = json_encode(['small' => $price]);
    }
    
    // Validate pizza category
    if (empty($pizza_category_id)) {
        $_SESSION['error_message'] = "Please select a pizza category";
        header("Location: menu.php?outlet=$outlet");
        exit;
    }
} else {
    // Regular items don't have sizes or pizza category
    $sizesJson = null;
    $pizza_category_id = null;
}

// Image upload
$image_url = $current_image;
if (!empty($_FILES['image']['name'])) {
    $uploadDir = __DIR__ . '/../images/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $fileExt = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
    $prefix = $item_type === 'pizza' ? 'pizza_' : 'item_';
    $filename = $prefix . time() . '_' . uniqid() . '.' . strtolower($fileExt);
    $targetPath = $uploadDir . $filename;
    
    // Validate image
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $fileType = mime_content_type($_FILES['image']['tmp_name']);
    
    if (in_array($fileType, $allowedTypes)) {
        if (move_uploaded_file($_FILES['image']['tmp_name'], $targetPath)) {
            // Delete old image if exists
            if (!empty($current_image) && file_exists($uploadDir . $current_image)) {
                unlink($uploadDir . $current_image);
            }
            $image_url = $filename;
        } else {
            $_SESSION['error_message'] = "Failed to upload image";
            header("Location: menu.php?outlet=$outlet");
            exit;
        }
    } else {
        $_SESSION['error_message'] = "Invalid image type. Allowed: JPG, PNG, GIF, WEBP";
        header("Location: menu.php?outlet=$outlet");
        exit;
    }
}

try {
    if ($id) {
        // Update existing item
        $stmt = $conn->prepare("
            UPDATE menu_items SET
            name = ?, description = ?, category = ?, price = ?, 
            sizes = ?, pizza_category_id = ?, image_url = ?, is_active = ?,
            updated_at = NOW()
            WHERE id = ? AND outlet = ?
        ");
        $stmt->execute([
            $name, $description, $category, $price,
            $sizesJson, $pizza_category_id, $image_url, $is_active,
            $id, $outlet
        ]);
        
        $successMessage = $item_type === 'pizza' ? "Pizza updated successfully!" : "Item updated successfully!";
        $_SESSION['success_message'] = $successMessage;
        
    } else {
        // Insert new item - need to get sort_order
        // Find max sort_order for this category
        $sortStmt = $conn->prepare("
            SELECT MAX(sort_order) as max_order 
            FROM menu_items 
            WHERE outlet = ? AND category = ?
        ");
        $sortStmt->execute([$outlet, $category]);
        $sortResult = $sortStmt->fetch(PDO::FETCH_ASSOC);
        $sort_order = ($sortResult['max_order'] ?? 0) + 1;
        
        $stmt = $conn->prepare("
            INSERT INTO menu_items 
            (name, description, category, price, sizes, pizza_category_id, 
             image_url, is_active, outlet, sort_order, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $name, $description, $category, $price,
            $sizesJson, $pizza_category_id, $image_url, $is_active,
            $outlet, $sort_order
        ]);
        
        $successMessage = $item_type === 'pizza' ? "Pizza added successfully!" : "Item added successfully!";
        $_SESSION['success_message'] = $successMessage;
    }
    
    // Clear cache
    $cacheFile = __DIR__ . "/../cache/menu_cache_{$outlet}.json";
    if (file_exists($cacheFile)) {
        unlink($cacheFile);
    }
    
    // Also clear pizza cache if pizza item
    if ($item_type === 'pizza') {
        $pizzaCacheFile = __DIR__ . "/../cache/pizza_cache_{$outlet}.json";
        if (file_exists($pizzaCacheFile)) {
            unlink($pizzaCacheFile);
        }
    }
    
} catch (Exception $e) {
    $errorMessage = $item_type === 'pizza' ? "pizza" : "item";
    $_SESSION['error_message'] = "Error saving {$errorMessage}: " . $e->getMessage();
    error_log("Menu save error: " . $e->getMessage());
}

header("Location: menu.php?outlet=$outlet");
exit;
?>