<?php
// admin/order_details.php - Order Details

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

if (!isset($_GET['id'])) {
    header('Location: orders.php');
    exit;
}

$order_id = $_GET['id'];
$database = new Database();
$conn = $database->getConnection();

// Get order details
$stmt = $conn->prepare("SELECT * FROM orders WHERE id = :id");
$stmt->execute([':id' => $order_id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    header('Location: orders.php');
    exit;
}

// Get order items
$stmt = $conn->prepare("SELECT * FROM order_items WHERE order_id = :id ORDER BY id");
$stmt->execute([':id' => $order_id]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get M-Pesa transaction if exists
$mpesa_transaction = null;
if ($order['mpesa_checkout_id']) {
    $stmt = $conn->prepare("SELECT * FROM mpesa_transactions WHERE checkout_request_id = :checkout_id ORDER BY id DESC LIMIT 1");
    $stmt->execute([':checkout_id' => $order['mpesa_checkout_id']]);
    $mpesa_transaction = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    $status = $_POST['status'];
    $notes = $_POST['notes'] ?? '';
    
    $stmt = $conn->prepare("UPDATE orders SET status = :status, payment_notes = :notes, updated_at = NOW() WHERE id = :id");
    $stmt->execute([
        ':status' => $status,
        ':notes' => $notes,
        ':id' => $order_id
    ]);
    
    // Update order variable
    $order['status'] = $status;
    $order['payment_notes'] = $notes;
    
    // Log activity
    $auth->logActivity($_SESSION['admin_id'], 'order_update', "Updated order #$order_id status to $status");
    
    $success = "Order status updated successfully!";
}

// Handle add note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_note'])) {
    $note = trim($_POST['note']);
    
    if (!empty($note)) {
        $current_notes = $order['payment_notes'] ?? '';
        $new_notes = $current_notes . "\n[" . date('Y-m-d H:i:s') . "] " . $note;
        
        $stmt = $conn->prepare("UPDATE orders SET payment_notes = :notes, updated_at = NOW() WHERE id = :id");
        $stmt->execute([
            ':notes' => $new_notes,
            ':id' => $order_id
        ]);
        
        $order['payment_notes'] = $new_notes;
        $note_success = "Note added successfully!";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order #<?php echo $order['order_number']; ?> - SUSHI-BAR Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        .order-header {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 15px rgba(0,0,0,0.08);
        }
        
        .order-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
        }
        
        .badge-pending { background: #ffc107; color: #212529; }
        .badge-processing { background: #17a2b8; color: white; }
        .badge-completed { background: #28a745; color: white; }
        .badge-paid { background: #2e7d32; color: white; }
        .badge-cancelled { background: #dc3545; color: white; }
        
        .item-row {
            border-bottom: 1px solid #eee;
            padding: 10px 0;
        }
        
        .item-row:last-child {
            border-bottom: none;
        }
        
        .total-row {
            background: #f8f9fa;
            font-weight: bold;
            padding: 15px;
            border-radius: 5px;
            margin-top: 10px;
        }
        
        .mpesa-details {
            background: #e8f5e9;
            border-left: 4px solid #2e7d32;
            padding: 15px;
            border-radius: 0 5px 5px 0;
        }
        
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        
        .timeline::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #dee2e6;
        }
        
        .timeline-item {
            position: relative;
            margin-bottom: 20px;
        }
        
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -33px;
            top: 5px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #e53935;
            border: 2px solid white;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-8">
                <!-- Order Header -->
                <div class="order-header">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h3>Order #<?php echo htmlspecialchars($order['order_number']); ?></h3>
                            <p class="text-muted mb-0">
                                <i class="fas fa-calendar"></i> 
                                <?php echo date('F j, Y g:i A', strtotime($order['created_at'])); ?>
                            </p>
                        </div>
                        <div class="text-end">
                            <span class="badge <?php echo 'badge-' . $order['status']; ?> fs-6">
                                <?php echo ucfirst($order['status']); ?>
                            </span>
                            <p class="mt-2 mb-0">
                                <strong>KES <?php echo number_format($order['total_amount'], 2); ?></strong>
                            </p>
                        </div>
                    </div>
                    
                    <!-- Status Update Form -->
                    <form method="POST" action="" class="mt-4">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Update Status</label>
                                <select name="status" class="form-select">
                                    <option value="pending" <?php echo $order['status'] === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                    <option value="processing" <?php echo $order['status'] === 'processing' ? 'selected' : ''; ?>>Processing</option>
                                    <option value="completed" <?php echo $order['status'] === 'completed' ? 'selected' : ''; ?>>Completed</option>
                                    <option value="paid" <?php echo $order['status'] === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                    <option value="cancelled" <?php echo $order['status'] === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" name="update_status" class="btn btn-primary w-100">
                                    Update Status
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Order Items -->
                <div class="order-card">
                    <h5 class="mb-4">Order Items</h5>
                    
                    <?php foreach ($items as $item): ?>
                    <div class="item-row row">
                        <div class="col-6">
                            <strong><?php echo htmlspecialchars($item['item_name']); ?></strong>
                        </div>
                        <div class="col-2 text-center">
                            <?php echo $item['quantity']; ?> ×
                        </div>
                        <div class="col-2 text-end">
                            KES <?php echo number_format($item['item_price'], 2); ?>
                        </div>
                        <div class="col-2 text-end">
                            <strong>KES <?php echo number_format($item['subtotal'], 2); ?></strong>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    
                    <!-- Order Totals -->
                    <div class="total-row">
                        <div class="row">
                            <div class="col-8 text-end">
                                Subtotal:
                            </div>
                            <div class="col-4 text-end">
                                KES <?php echo number_format($order['total_amount'] - $order['delivery_fee'], 2); ?>
                            </div>
                        </div>
                        
                        <?php if ($order['delivery_fee'] > 0): ?>
                        <div class="row">
                            <div class="col-8 text-end">
                                Delivery Fee:
                            </div>
                            <div class="col-4 text-end">
                                KES <?php echo number_format($order['delivery_fee'], 2); ?>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <div class="row" style="font-size: 1.1rem;">
                            <div class="col-8 text-end">
                                <strong>Total:</strong>
                            </div>
                            <div class="col-4 text-end">
                                <strong>KES <?php echo number_format($order['total_amount'], 2); ?></strong>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Customer Information -->
                <div class="order-card">
                    <h5 class="mb-4">Customer Information</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Name:</strong><br>
                            <?php echo htmlspecialchars($order['customer_name']); ?></p>
                            
                            <p><strong>Phone:</strong><br>
                            <?php echo htmlspecialchars($order['customer_phone']); ?></p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Delivery Location:</strong><br>
                            <?php echo htmlspecialchars($order['delivery_location']); ?></p>
                            
                            <p><strong>Delivery Address:</strong><br>
                            <?php echo nl2br(htmlspecialchars($order['customer_address'])); ?></p>
                        </div>
                    </div>
                </div>
                
                <!-- Notes -->
                <div class="order-card">
                    <h5 class="mb-4">Notes</h5>
                    <form method="POST" action="" class="mb-3">
                        <div class="input-group">
                            <textarea name="note" class="form-control" placeholder="Add a note..." rows="2"></textarea>
                            <button type="submit" name="add_note" class="btn btn-primary">
                                <i class="fas fa-plus"></i> Add
                            </button>
                        </div>
                    </form>
                    
                    <?php if (!empty($order['payment_notes'])): ?>
                    <div class="timeline">
                        <?php 
                        $notes = explode("\n", $order['payment_notes']);
                        foreach ($notes as $note):
                            if (trim($note)):
                        ?>
                        <div class="timeline-item">
                            <small class="text-muted"><?php echo htmlspecialchars($note); ?></small>
                        </div>
                        <?php 
                            endif;
                        endforeach; 
                        ?>
                    </div>
                    <?php else: ?>
                    <p class="text-muted">No notes added yet.</p>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Sidebar -->
            <div class="col-md-4">
                <!-- Payment Information -->
                <div class="order-card">
                    <h5 class="mb-4">Payment Information</h5>
                    
                    <p><strong>Payment Method:</strong><br>
                    <span class="badge bg-<?php echo $order['payment_method'] === 'mpesa' ? 'success' : 'info'; ?>">
                        <?php echo strtoupper($order['payment_method']); ?>
                    </span></p>
                    
                    <?php if ($order['payment_method'] === 'mpesa' && $mpesa_transaction): ?>
                    <div class="mpesa-details mt-3">
                        <h6>M-Pesa Details</h6>
                        <p class="mb-1"><strong>Receipt:</strong> <?php echo $mpesa_transaction['mpesa_receipt_number'] ?? 'N/A'; ?></p>
                        <p class="mb-1"><strong>Amount:</strong> KES <?php echo number_format($mpesa_transaction['amount'] ?? 0, 2); ?></p>
                        <p class="mb-1"><strong>Status:</strong> 
                            <span class="badge bg-<?php echo $mpesa_transaction['status'] === 'success' ? 'success' : 'danger'; ?>">
                                <?php echo ucfirst($mpesa_transaction['status'] ?? 'pending'); ?>
                            </span>
                        </p>
                        <?php if ($mpesa_transaction['transaction_date']): ?>
                        <p class="mb-0"><strong>Date:</strong> <?php echo $mpesa_transaction['transaction_date']; ?></p>
                        <?php endif; ?>
                    </div>
                    <?php elseif ($order['payment_method'] === 'mpesa'): ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        No M-Pesa transaction details found.
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Order Timeline -->
                <div class="order-card">
                    <h5 class="mb-4">Order Timeline</h5>
                    <div class="timeline">
                        <div class="timeline-item">
                            <strong>Order Created</strong>
                            <p class="text-muted mb-0"><?php echo date('M j, g:i A', strtotime($order['created_at'])); ?></p>
                        </div>
                        
                        <?php if ($order['updated_at'] !== $order['created_at']): ?>
                        <div class="timeline-item">
                            <strong>Last Updated</strong>
                            <p class="text-muted mb-0"><?php echo date('M j, g:i A', strtotime($order['updated_at'])); ?></p>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($order['payment_date']): ?>
                        <div class="timeline-item">
                            <strong>Payment Received</strong>
                            <p class="text-muted mb-0"><?php echo date('M j, g:i A', strtotime($order['payment_date'])); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="order-card">
                    <h5 class="mb-4">Quick Actions</h5>
                    <div class="d-grid gap-2">
                        <a href="order_print.php?id=<?php echo $order['id']; ?>" 
                           target="_blank"
                           class="btn btn-outline-primary">
                            <i class="fas fa-print"></i> Print Order
                        </a>
                        
                        <a href="order_edit.php?id=<?php echo $order['id']; ?>" 
                           class="btn btn-outline-success">
                            <i class="fas fa-edit"></i> Edit Order
                        </a>
                        
                        <?php if ($order['customer_phone']): ?>
                        <a href="https://wa.me/<?php echo $order['customer_phone']; ?>?text=Hello%20<?php echo urlencode($order['customer_name']); ?>%2C%20this%20is%20SUSHI-BAR%20regarding%20your%20order%20%23<?php echo $order['order_number']; ?>" 
                           target="_blank"
                           class="btn btn-outline-success" style="background: #25D366; color: white; border-color: #25D366;">
                            <i class="fab fa-whatsapp"></i> WhatsApp Customer
                        </a>
                        <?php endif; ?>
                        
                        <a href="orders.php" class="btn btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Back to Orders
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>