<?php
// admin/orders.php - Orders Management

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    $order_id = $_POST['order_id'];
    $status = $_POST['status'];
    $notes = $_POST['notes'] ?? '';
    
    $stmt = $conn->prepare("UPDATE orders SET status = :status, payment_notes = :notes, updated_at = NOW() WHERE id = :id");
    $stmt->execute([
        ':status' => $status,
        ':notes' => $notes,
        ':id' => $order_id
    ]);
    
    // Log activity
    $auth->logActivity($_SESSION['admin_id'], 'order_update', "Updated order #$order_id status to $status");
    
    $success = "Order status updated successfully!";
}

// Handle search and filters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build query
$query = "SELECT * FROM orders WHERE 1=1";
$params = [];

if (!empty($search)) {
    $query .= " AND (order_number LIKE :search OR customer_name LIKE :search OR customer_phone LIKE :search)";
    $params[':search'] = "%$search%";
}

if (!empty($status_filter)) {
    $query .= " AND status = :status";
    $params[':status'] = $status_filter;
}

if (!empty($date_from)) {
    $query .= " AND DATE(created_at) >= :date_from";
    $params[':date_from'] = $date_from;
}

if (!empty($date_to)) {
    $query .= " AND DATE(created_at) <= :date_to";
    $params[':date_to'] = $date_to;
}

$query .= " ORDER BY created_at DESC";

// Pagination
$page = $_GET['page'] ?? 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Count total
$count_query = str_replace('SELECT *', 'SELECT COUNT(*) as total', $query);
$stmt = $conn->prepare($count_query);
$stmt->execute($params);
$total_rows = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
$total_pages = ceil($total_rows / $limit);

// Get orders
$query .= " LIMIT $limit OFFSET $offset";
$stmt = $conn->prepare($query);
$stmt->execute($params);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get status counts for filters
$status_counts = [];
$stmt = $conn->query("SELECT status, COUNT(*) as count FROM orders GROUP BY status");
$status_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

foreach ($status_data as $row) {
    $status_counts[$row['status']] = $row['count'];
}

// Get all statuses for filter dropdown
$statuses = ['pending', 'processing', 'completed', 'paid', 'cancelled', 'failed'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Orders Management - SUSHI-BAR Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        :root {
            --primary: #e53935;
            --primary-dark: #c62828;
        }
        
        .order-status-select {
            padding: 5px 10px;
            border-radius: 5px;
            border: 1px solid #ddd;
            font-size: 0.9rem;
        }
        
        .badge-pending { background: #ffc107; color: #212529; }
        .badge-processing { background: #17a2b8; color: white; }
        .badge-completed { background: #28a745; color: white; }
        .badge-paid { background: #2e7d32; color: white; }
        .badge-cancelled { background: #dc3545; color: white; }
        .badge-failed { background: #6c757d; color: white; }
        
        .filter-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.08);
        }
        
        .status-filter {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            margin-bottom: 15px;
        }
        
        .status-filter .btn {
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 0.9rem;
        }
        
        .order-row:hover {
            background-color: #f8f9fa;
        }
        
        .pagination .page-link {
            color: var(--primary);
        }
        
        .pagination .page-item.active .page-link {
            background-color: var(--primary);
            border-color: var(--primary);
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Orders Management</h2>
                    <div>
                        <a href="new_order.php" class="btn btn-primary">
                            <i class="fas fa-plus"></i> New Order
                        </a>
                        <a href="reports.php" class="btn btn-outline-primary">
                            <i class="fas fa-download"></i> Export
                        </a>
                    </div>
                </div>
                
                <?php if (isset($success)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle"></i> <?php echo $success; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- Filters -->
                <div class="filter-card">
                    <form method="GET" action="" class="row g-3">
                        <div class="col-md-3">
                            <input type="text" 
                                   class="form-control" 
                                   name="search" 
                                   placeholder="Search orders..."
                                   value="<?php echo htmlspecialchars($search); ?>">
                        </div>
                        
                        <div class="col-md-2">
                            <select class="form-select" name="status">
                                <option value="">All Status</option>
                                <?php foreach ($statuses as $status): ?>
                                <option value="<?php echo $status; ?>" 
                                    <?php echo $status_filter === $status ? 'selected' : ''; ?>>
                                    <?php echo ucfirst($status); ?> 
                                    (<?php echo $status_counts[$status] ?? 0; ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" 
                                   class="form-control" 
                                   name="date_from" 
                                   value="<?php echo htmlspecialchars($date_from); ?>"
                                   placeholder="From Date">
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" 
                                   class="form-control" 
                                   name="date_to" 
                                   value="<?php echo htmlspecialchars($date_to); ?>"
                                   placeholder="To Date">
                        </div>
                        
                        <div class="col-md-3">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-search"></i> Filter
                            </button>
                        </div>
                    </form>
                    
                    <!-- Status Quick Filters -->
                    <div class="status-filter mt-3">
                        <a href="?status=" class="btn btn-outline-secondary">
                            All (<?php echo array_sum($status_counts); ?>)
                        </a>
                        <?php foreach ($statuses as $status): 
                            $count = $status_counts[$status] ?? 0;
                            if ($count > 0): ?>
                            <a href="?status=<?php echo $status; ?>" 
                               class="btn btn-outline-<?php 
                                   echo $status === 'pending' ? 'warning' : 
                                        ($status === 'processing' ? 'info' : 
                                        ($status === 'completed' ? 'success' : 
                                        ($status === 'paid' ? 'success' : 
                                        ($status === 'cancelled' ? 'danger' : 'secondary')))); ?>">
                                <?php echo ucfirst($status); ?> (<?php echo $count; ?>)
                            </a>
                            <?php endif;
                        endforeach; ?>
                    </div>
                </div>
                
                <!-- Orders Table -->
                <div class="card">
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Order #</th>
                                        <th>Customer</th>
                                        <th>Delivery</th>
                                        <th>Items</th>
                                        <th>Amount</th>
                                        <th>Payment</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($orders as $order): 
                                        // Get order items count
                                        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM order_items WHERE order_id = :id");
                                        $stmt->execute([':id' => $order['id']]);
                                        $item_count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                                    ?>
                                    <tr class="order-row">
                                        <td>
                                            <strong><?php echo htmlspecialchars($order['order_number']); ?></strong>
                                            <?php if ($order['delivery_fee'] > 0): ?>
                                            <br><small class="text-muted">+KES <?php echo $order['delivery_fee']; ?> delivery</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($order['customer_name']); ?></div>
                                            <small class="text-muted"><?php echo htmlspecialchars($order['customer_phone']); ?></small>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($order['delivery_location']); ?>
                                            <br>
                                            <small class="text-muted"><?php echo substr($order['customer_address'], 0, 30); ?>...</small>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?php echo $item_count; ?> items</span>
                                        </td>
                                        <td>
                                            <strong>KES <?php echo number_format($order['total_amount'], 2); ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo $order['payment_method'] === 'mpesa' ? 'success' : 'info'; ?>">
                                                <?php echo strtoupper($order['payment_method']); ?>
                                            </span>
                                            <?php if ($order['mpesa_receipt']): ?>
                                            <br><small><?php echo $order['mpesa_receipt']; ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <form method="POST" action="" class="d-inline">
                                                <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                <select name="status" class="order-status-select" 
                                                        onchange="this.form.submit()">
                                                    <?php foreach ($statuses as $status): ?>
                                                    <option value="<?php echo $status; ?>" 
                                                        <?php echo $order['status'] === $status ? 'selected' : ''; ?>>
                                                        <?php echo ucfirst($status); ?>
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                                <input type="hidden" name="update_status" value="1">
                                            </form>
                                        </td>
                                        <td>
                                            <?php echo date('M j, g:i A', strtotime($order['created_at'])); ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="order_details.php?id=<?php echo $order['id']; ?>" 
                                                   class="btn btn-outline-primary" 
                                                   title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <a href="order_print.php?id=<?php echo $order['id']; ?>" 
                                                   target="_blank"
                                                   class="btn btn-outline-secondary" 
                                                   title="Print">
                                                    <i class="fas fa-print"></i>
                                                </a>
                                                <a href="order_edit.php?id=<?php echo $order['id']; ?>" 
                                                   class="btn btn-outline-success" 
                                                   title="Edit">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    
                                    <?php if (empty($orders)): ?>
                                    <tr>
                                        <td colspan="9" class="text-center py-4">
                                            <i class="fas fa-shopping-cart fa-2x text-muted mb-3"></i>
                                            <p class="text-muted">No orders found</p>
                                            <a href="new_order.php" class="btn btn-primary">Create New Order</a>
                                        </td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav class="mt-4">
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $page - 1; ?><?php echo !empty($search) ? "&search=$search" : ''; ?><?php echo !empty($status_filter) ? "&status=$status_filter" : ''; ?>">
                                Previous
                            </a>
                        </li>
                        
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?><?php echo !empty($search) ? "&search=$search" : ''; ?><?php echo !empty($status_filter) ? "&status=$status_filter" : ''; ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                        <?php endfor; ?>
                        
                        <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $page + 1; ?><?php echo !empty($search) ? "&search=$search" : ''; ?><?php echo !empty($status_filter) ? "&status=$status_filter" : ''; ?>">
                                Next
                            </a>
                        </li>
                    </ul>
                </nav>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-submit status change with notes
        document.querySelectorAll('.order-status-select').forEach(select => {
            select.addEventListener('change', function() {
                const orderId = this.form.querySelector('input[name="order_id"]').value;
                const status = this.value;
                
                // Show notes input for cancelled or failed status
                if (status === 'cancelled' || status === 'failed') {
                    const notes = prompt('Enter reason for ' + status + ' status:', '');
                    if (notes !== null) {
                        const notesInput = document.createElement('input');
                        notesInput.type = 'hidden';
                        notesInput.name = 'notes';
                        notesInput.value = notes;
                        this.form.appendChild(notesInput);
                        this.form.submit();
                    } else {
                        this.value = this.form.querySelector('option[selected]').value;
                    }
                } else {
                    this.form.submit();
                }
            });
        });
        
        // Auto-refresh orders every 60 seconds
        setInterval(function() {
            if (!document.hidden) {
                window.location.reload();
            }
        }, 60000);
    </script>
</body>
</html>