<?php
// admin/reports.php - Reports Management

session_start();
require_once __DIR__ . '/../config/database.php';
require_once 'auth.php';

$auth = new AdminAuth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$conn = $database->getConnection();

// Get outlet parameter
$outlet = $_GET['outlet'] ?? 'milano';
$report_type = $_GET['report'] ?? 'overview';
$date_from = $_GET['date_from'] ?? date('Y-m-01'); // Start of current month
$date_to = $_GET['date_to'] ?? date('Y-m-d');
$group_by = $_GET['group_by'] ?? 'day';
$category_filter = $_GET['category'] ?? '';

// Validate dates
if (!empty($date_from) && !empty($date_to) && strtotime($date_to) < strtotime($date_from)) {
    $date_to = $date_from;
}

// Initialize statistics arrays
$stats = [];
$revenue_data = [];
$order_data = [];
$top_items = [];
$top_customers = [];

try {
    // Overview Statistics
    if ($report_type === 'overview' || $report_type === 'dashboard') {
        // Total Revenue
        $query = "SELECT 
                    SUM(total_amount) as total_revenue,
                    COUNT(*) as total_orders,
                    AVG(total_amount) as avg_order_value,
                    COUNT(DISTINCT customer_id) as unique_customers
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $stats = $stmt->fetch(PDO::FETCH_ASSOC) ?? [];
        
        // Revenue by day/week/month
        $group_format = $group_by === 'month' ? '%Y-%m' : ($group_by === 'week' ? '%Y-%U' : '%Y-%m-%d');
        
        $query = "SELECT 
                    DATE_FORMAT(created_at, :format) as period,
                    SUM(total_amount) as revenue,
                    COUNT(*) as orders
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY period
                  ORDER BY period";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':format' => $group_format,
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $revenue_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Top selling items - FIXED: Assuming you have an order_items table
        $query = "SELECT 
                    mi.name,
                    mi.category,
                    SUM(oi.quantity) as quantity_sold,
                    SUM(oi.price * oi.quantity) as revenue
                  FROM order_items oi
                  JOIN orders o ON oi.order_id = o.id
                  JOIN menu_items mi ON oi.menu_item_id = mi.id
                  WHERE o.outlet = :outlet 
                    AND o.status IN ('completed', 'delivered', 'paid')
                    AND DATE(o.created_at) BETWEEN :date_from AND :date_to
                  GROUP BY mi.id
                  ORDER BY quantity_sold DESC
                  LIMIT 10";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $top_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Top customers
        $query = "SELECT 
                    c.name,
                    c.email,
                    c.phone,
                    COUNT(o.id) as order_count,
                    SUM(o.total_amount) as total_spent
                  FROM orders o
                  LEFT JOIN customers c ON o.customer_id = c.id
                  WHERE o.outlet = :outlet 
                    AND o.status IN ('completed', 'delivered', 'paid')
                    AND DATE(o.created_at) BETWEEN :date_from AND :date_to
                  GROUP BY o.customer_id
                  ORDER BY total_spent DESC
                  LIMIT 10";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $top_customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Order status distribution
        $query = "SELECT 
                    status,
                    COUNT(*) as count,
                    SUM(total_amount) as revenue
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY status";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $order_statuses = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Payment method distribution
        $query = "SELECT 
                    payment_method,
                    COUNT(*) as count,
                    SUM(total_amount) as revenue
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY payment_method";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $payment_methods = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Sales by Category - FIXED: Using order_items table
    if ($report_type === 'sales_by_category') {
        $query = "SELECT 
                    mi.category,
                    SUM(oi.quantity) as quantity_sold,
                    SUM(oi.price * oi.quantity) as revenue,
                    AVG(oi.price) as avg_price
                  FROM order_items oi
                  JOIN orders o ON oi.order_id = o.id
                  JOIN menu_items mi ON oi.menu_item_id = mi.id
                  WHERE o.outlet = :outlet 
                    AND o.status IN ('completed', 'delivered', 'paid')
                    AND DATE(o.created_at) BETWEEN :date_from AND :date_to";
        
        if (!empty($category_filter)) {
            $query .= " AND mi.category = :category";
        }
        
        $query .= " GROUP BY mi.category ORDER BY revenue DESC";
        
        $stmt = $conn->prepare($query);
        $params = [
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ];
        
        if (!empty($category_filter)) {
            $params[':category'] = $category_filter;
        }
        
        $stmt->execute($params);
        $category_sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Customer Reports
    if ($report_type === 'customers') {
        $query = "SELECT 
                    c.*,
                    COUNT(o.id) as total_orders,
                    SUM(o.total_amount) as total_revenue,
                    MAX(o.created_at) as last_order_date,
                    MIN(o.created_at) as first_order_date,
                    AVG(o.total_amount) as avg_order_value
                  FROM customers c
                  LEFT JOIN orders o ON c.id = o.customer_id AND o.outlet = :outlet
                  GROUP BY c.id
                  ORDER BY total_revenue DESC";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([':outlet' => $outlet]);
        $customer_reports = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Hourly/Daily Performance
    if ($report_type === 'hourly') {
        $query = "SELECT 
                    HOUR(created_at) as hour,
                    DAYNAME(created_at) as day_name,
                    COUNT(*) as order_count,
                    SUM(total_amount) as revenue,
                    AVG(total_amount) as avg_order_value
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY hour, day_name
                  ORDER BY hour";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $hourly_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Group by day of week
        $query = "SELECT 
                    DAYNAME(created_at) as day_name,
                    DAYOFWEEK(created_at) as day_num,
                    COUNT(*) as order_count,
                    SUM(total_amount) as revenue,
                    AVG(total_amount) as avg_order_value
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY day_name, day_num
                  ORDER BY day_num";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $daily_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Delivery Performance
    if ($report_type === 'delivery') {
        $query = "SELECT 
                    delivery_location,
                    COUNT(*) as order_count,
                    SUM(total_amount) as revenue,
                    AVG(total_amount) as avg_order_value,
                    SUM(delivery_fee) as total_delivery_fees,
                    AVG(TIMESTAMPDIFF(MINUTE, created_at, updated_at)) as avg_delivery_time
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered')
                    AND delivery_location IS NOT NULL
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY delivery_location
                  ORDER BY order_count DESC";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $delivery_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Financial Summary
    if ($report_type === 'financial') {
        // Revenue by payment method
        $query = "SELECT 
                    payment_method,
                    COUNT(*) as transaction_count,
                    SUM(total_amount) as revenue,
                    SUM(CASE WHEN payment_status = 'paid' THEN total_amount ELSE 0 END) as collected_amount
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY payment_method";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $financial_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Daily revenue trend
        $query = "SELECT 
                    DATE(created_at) as date,
                    SUM(total_amount) as daily_revenue,
                    COUNT(*) as daily_orders,
                    SUM(delivery_fee) as daily_delivery_fees
                  FROM orders 
                  WHERE outlet = :outlet 
                    AND status IN ('completed', 'delivered', 'paid')
                    AND DATE(created_at) BETWEEN :date_from AND :date_to
                  GROUP BY DATE(created_at)
                  ORDER BY date";
        
        $stmt = $conn->prepare($query);
        $stmt->execute([
            ':outlet' => $outlet,
            ':date_from' => $date_from,
            ':date_to' => $date_to
        ]);
        $daily_financial = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
} catch (Exception $e) {
    error_log("Error generating reports: " . $e->getMessage());
    $error_message = "Error generating reports: " . $e->getMessage();
}

// Get available categories for filter
$query = "SELECT DISTINCT category FROM menu_items WHERE outlet = :outlet AND category IS NOT NULL ORDER BY category";
$stmt = $conn->prepare($query);
$stmt->execute([':outlet' => $outlet]);
$available_categories = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);

// Get month names for select
$months = [
    '01' => 'January', '02' => 'February', '03' => 'March', '04' => 'April',
    '05' => 'May', '06' => 'June', '07' => 'July', '08' => 'August',
    '09' => 'September', '10' => 'October', '11' => 'November', '12' => 'December'
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        :root {
            --primary: #D35400;
            --primary-dark: #A04000;
            --secondary: #27AE60;
            --info: #17a2b8;
            --warning: #ffc107;
            --danger: #dc3545;
        }
        
        .stats-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
            border-left: 4px solid var(--primary);
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        .stats-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.1);
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            margin-bottom: 15px;
        }
        
        .report-card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
            border: 1px solid #eaeaea;
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 20px;
        }
        
        .table-responsive {
            max-height: 400px;
            overflow-y: auto;
        }
        
        .filter-bar {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 25px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .nav-tabs .nav-link {
            border: none;
            color: #6c757d;
            font-weight: 500;
            padding: 10px 20px;
        }
        
        .nav-tabs .nav-link.active {
            color: var(--primary);
            border-bottom: 3px solid var(--primary);
            background: transparent;
        }
        
        .badge-outline {
            background: transparent;
            border: 1px solid;
            font-weight: 500;
        }
        
        .progress-thin {
            height: 8px;
            border-radius: 4px;
        }
        
        .trend-up {
            color: var(--secondary);
        }
        
        .trend-down {
            color: var(--danger);
        }
        
        .outlet-badge {
            font-size: 0.8rem;
            padding: 4px 10px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .data-table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background: #f8f9fa;
        }
        
        .export-btn {
            border-radius: 8px;
            padding: 8px 20px;
            font-weight: 500;
        }
        
        .date-range {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .avatar-placeholder {
            width: 36px;
            height: 36px;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container-fluid mt-4">
        <div class="row">
            <div class="col-md-12">
                <!-- Outlet Selector -->
                <div class="mb-4">
                    <div class="btn-group" role="group">
                        <a href="?outlet=milano&report=<?php echo $report_type; ?>" 
                           class="btn <?php echo $outlet === 'milano' ? 'btn-warning' : 'btn-outline-warning'; ?>">
                            <i class="fas fa-pizza-slice"></i> Milano Pizza
                        </a>
                        <a href="?outlet=sushi&report=<?php echo $report_type; ?>" 
                           class="btn <?php echo $outlet === 'sushi' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-fish"></i> Sushi Bar
                        </a>
                    </div>
                </div>
                
                <?php if (isset($error_message)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error_message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- Report Tabs -->
                <ul class="nav nav-tabs mb-4" id="reportTabs">
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'overview' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=overview">
                            <i class="fas fa-chart-line me-2"></i>Overview
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'sales_by_category' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=sales_by_category">
                            <i class="fas fa-tags me-2"></i>Sales by Category
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'customers' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=customers">
                            <i class="fas fa-users me-2"></i>Customer Reports
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'hourly' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=hourly">
                            <i class="fas fa-clock me-2"></i>Hourly/Daily
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'delivery' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=delivery">
                            <i class="fas fa-truck me-2"></i>Delivery
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $report_type === 'financial' ? 'active' : ''; ?>" 
                           href="?outlet=<?php echo $outlet; ?>&report=financial">
                            <i class="fas fa-money-bill-wave me-2"></i>Financial
                        </a>
                    </li>
                </ul>
                
                <!-- Filter Bar -->
                <div class="filter-bar">
                    <form method="GET" action="" class="row g-3">
                        <input type="hidden" name="outlet" value="<?php echo $outlet; ?>">
                        <input type="hidden" name="report" value="<?php echo $report_type; ?>">
                        
                        <div class="col-md-3">
                            <label class="form-label">Date From</label>
                            <input type="date" class="form-control" name="date_from" 
                                   value="<?php echo htmlspecialchars($date_from); ?>" required>
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Date To</label>
                            <input type="date" class="form-control" name="date_to" 
                                   value="<?php echo htmlspecialchars($date_to); ?>" required>
                        </div>
                        
                        <div class="col-md-2">
                            <label class="form-label">Group By</label>
                            <select class="form-select" name="group_by">
                                <option value="day" <?php echo $group_by === 'day' ? 'selected' : ''; ?>>Daily</option>
                                <option value="week" <?php echo $group_by === 'week' ? 'selected' : ''; ?>>Weekly</option>
                                <option value="month" <?php echo $group_by === 'month' ? 'selected' : ''; ?>>Monthly</option>
                            </select>
                        </div>
                        
                        <?php if ($report_type === 'sales_by_category'): ?>
                        <div class="col-md-2">
                            <label class="form-label">Category</label>
                            <select class="form-select" name="category">
                                <option value="">All Categories</option>
                                <?php foreach ($available_categories as $category): ?>
                                <option value="<?php echo htmlspecialchars($category); ?>"
                                    <?php echo $category_filter === $category ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars(ucfirst($category)); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <div class="col-md-2 d-flex align-items-end">
                            <div class="d-grid gap-2 w-100">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-filter me-2"></i>Apply Filters
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Date Range Display -->
                <div class="date-range">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <h5 class="mb-0">
                                <i class="fas fa-calendar-alt me-2 text-primary"></i>
                                Report Period: <?php echo date('M j, Y', strtotime($date_from)); ?> 
                                to <?php echo date('M j, Y', strtotime($date_to)); ?>
                            </h5>
                        </div>
                        <div class="col-md-6 text-end">
                            <button class="btn btn-outline-primary export-btn me-2" onclick="exportReport()">
                                <i class="fas fa-download me-2"></i>Export Report
                            </button>
                            <button class="btn btn-outline-secondary export-btn" onclick="printReport()">
                                <i class="fas fa-print me-2"></i>Print
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Overview Report -->
                <?php if ($report_type === 'overview'): ?>
                <div class="row mb-4">
                    <!-- Key Statistics -->
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e8f5e9; color: var(--secondary);">
                                <i class="fas fa-money-bill-wave"></i>
                            </div>
                            <h3 class="mb-1">KES <?php echo number_format($stats['total_revenue'] ?? 0, 2); ?></h3>
                            <p class="text-muted mb-0">Total Revenue</p>
                            <small class="text-success">
                                <i class="fas fa-chart-line me-1"></i>
                                From <?php echo number_format($stats['total_orders'] ?? 0); ?> orders
                            </small>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e3f2fd; color: var(--primary);">
                                <i class="fas fa-shopping-cart"></i>
                            </div>
                            <h3 class="mb-1"><?php echo number_format($stats['total_orders'] ?? 0); ?></h3>
                            <p class="text-muted mb-0">Total Orders</p>
                            <small class="text-primary">
                                <i class="fas fa-users me-1"></i>
                                <?php echo number_format($stats['unique_customers'] ?? 0); ?> customers
                            </small>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #fff3e0; color: var(--warning);">
                                <i class="fas fa-chart-bar"></i>
                            </div>
                            <h3 class="mb-1">KES <?php echo number_format($stats['avg_order_value'] ?? 0, 2); ?></h3>
                            <p class="text-muted mb-0">Average Order Value</p>
                            <small class="text-warning">
                                <i class="fas fa-percentage me-1"></i>
                                Per order average
                            </small>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #f3e5f5; color: #7b1fa2;">
                                <i class="fas fa-user-check"></i>
                            </div>
                            <h3 class="mb-1"><?php echo number_format($stats['unique_customers'] ?? 0); ?></h3>
                            <p class="text-muted mb-0">Unique Customers</p>
                            <small class="text-info">
                                <i class="fas fa-retweet me-1"></i>
                                Repeat customers
                            </small>
                        </div>
                    </div>
                </div>
                
                <!-- Charts Row -->
                <div class="row mb-4">
                    <!-- Revenue Chart -->
                    <div class="col-md-8">
                        <div class="report-card">
                            <h5 class="mb-3">
                                <i class="fas fa-chart-line me-2 text-primary"></i>
                                Revenue Trend (<?php echo ucfirst($group_by); ?>)
                            </h5>
                            <div class="chart-container">
                                <canvas id="revenueChart"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Order Status Distribution -->
                    <div class="col-md-4">
                        <div class="report-card">
                            <h5 class="mb-3">
                                <i class="fas fa-chart-pie me-2 text-primary"></i>
                                Order Status Distribution
                            </h5>
                            <div class="chart-container">
                                <canvas id="statusChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Tables Row -->
                <div class="row">
                    <!-- Top Selling Items -->
                    <div class="col-md-6">
                        <div class="report-card">
                            <h5 class="mb-3">
                                <i class="fas fa-star me-2 text-warning"></i>
                                Top Selling Items
                            </h5>
                            <div class="table-responsive">
                                <table class="table table-hover data-table">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th>Category</th>
                                            <th>Quantity</th>
                                            <th>Revenue</th>
                                            <th>Avg Price</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (!empty($top_items)): ?>
                                        <?php foreach ($top_items as $item): 
                                            $avg_price = ($item['quantity_sold'] > 0) ? $item['revenue'] / $item['quantity_sold'] : 0;
                                        ?>
                                        <tr>
                                            <td>
                                                <strong><?php echo htmlspecialchars($item['name']); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-light text-dark">
                                                    <?php echo htmlspecialchars($item['category']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo $item['quantity_sold']; ?></td>
                                            <td class="text-success">
                                                <strong>KES <?php echo number_format($item['revenue'], 2); ?></strong>
                                            </td>
                                            <td>
                                                KES <?php echo number_format($avg_price, 2); ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                        <?php else: ?>
                                        <tr>
                                            <td colspan="5" class="text-center text-muted">No items sold in this period</td>
                                        </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Top Customers -->
                    <div class="col-md-6">
                        <div class="report-card">
                            <h5 class="mb-3">
                                <i class="fas fa-crown me-2 text-warning"></i>
                                Top Customers
                            </h5>
                            <div class="table-responsive">
                                <table class="table table-hover data-table">
                                    <thead>
                                        <tr>
                                            <th>Customer</th>
                                            <th>Orders</th>
                                            <th>Total Spent</th>
                                            <th>Avg Order</th>
                                            <th>Contact</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (!empty($top_customers)): ?>
                                        <?php foreach ($top_customers as $customer): 
                                            $avg_order = ($customer['order_count'] > 0) ? $customer['total_spent'] / $customer['order_count'] : 0;
                                        ?>
                                        <tr>
                                            <td>
                                                <strong><?php echo htmlspecialchars($customer['name'] ?? 'Unknown'); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-primary"><?php echo $customer['order_count']; ?></span>
                                            </td>
                                            <td class="text-success">
                                                <strong>KES <?php echo number_format($customer['total_spent'], 2); ?></strong>
                                            </td>
                                            <td>
                                                KES <?php echo number_format($avg_order, 2); ?>
                                            </td>
                                            <td>
                                                <?php if (!empty($customer['phone'])): ?>
                                                <small><?php echo htmlspecialchars($customer['phone']); ?></small>
                                                <?php elseif (!empty($customer['email'])): ?>
                                                <small><?php echo htmlspecialchars($customer['email']); ?></small>
                                                <?php else: ?>
                                                <small class="text-muted">No contact info</small>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                        <?php else: ?>
                                        <tr>
                                            <td colspan="5" class="text-center text-muted">No customer data found</td>
                                        </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Payment Method Distribution -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="report-card">
                            <h5 class="mb-3">
                                <i class="fas fa-credit-card me-2 text-primary"></i>
                                Payment Method Distribution
                            </h5>
                            <div class="chart-container">
                                <canvas id="paymentChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- JavaScript for Charts -->
                <script>
                    // Revenue Chart
                    const revenueCtx = document.getElementById('revenueChart').getContext('2d');
                    const revenueChart = new Chart(revenueCtx, {
                        type: 'line',
                        data: {
                            labels: [<?php 
                                foreach ($revenue_data as $data) {
                                    echo "'" . $data['period'] . "',";
                                }
                            ?>],
                            datasets: [{
                                label: 'Revenue',
                                data: [<?php 
                                    foreach ($revenue_data as $data) {
                                        echo $data['revenue'] . ",";
                                    }
                                ?>],
                                borderColor: '#D35400',
                                backgroundColor: 'rgba(211, 84, 0, 0.1)',
                                tension: 0.4,
                                fill: true
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    display: true,
                                    position: 'top',
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    ticks: {
                                        callback: function(value) {
                                            return 'KES ' + value.toLocaleString();
                                        }
                                    }
                                }
                            }
                        }
                    });
                    
                    // Status Distribution Chart
                    const statusCtx = document.getElementById('statusChart').getContext('2d');
                    const statusChart = new Chart(statusCtx, {
                        type: 'doughnut',
                        data: {
                            labels: [<?php 
                                foreach ($order_statuses as $status) {
                                    echo "'" . ucfirst($status['status']) . "',";
                                }
                            ?>],
                            datasets: [{
                                data: [<?php 
                                    foreach ($order_statuses as $status) {
                                        echo $status['count'] . ",";
                                    }
                                ?>],
                                backgroundColor: [
                                    '#D35400', '#27AE60', '#3498DB', '#F39C12', '#E74C3C', '#9B59B6'
                                ]
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'right',
                                }
                            }
                        }
                    });
                    
                    // Payment Method Chart
                    const paymentCtx = document.getElementById('paymentChart').getContext('2d');
                    const paymentChart = new Chart(paymentCtx, {
                        type: 'bar',
                        data: {
                            labels: [<?php 
                                foreach ($payment_methods as $method) {
                                    echo "'" . ucfirst($method['payment_method']) . "',";
                                }
                            ?>],
                            datasets: [{
                                label: 'Revenue',
                                data: [<?php 
                                    foreach ($payment_methods as $method) {
                                        echo $method['revenue'] . ",";
                                    }
                                ?>],
                                backgroundColor: '#D35400',
                                borderColor: '#A04000',
                                borderWidth: 1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    display: false
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    ticks: {
                                        callback: function(value) {
                                            return 'KES ' + value.toLocaleString();
                                        }
                                    }
                                }
                            }
                        }
                    });
                </script>
                <?php endif; ?>
                
                <!-- Sales by Category Report -->
                <?php if ($report_type === 'sales_by_category'): ?>
                <div class="report-card">
                    <h5 class="mb-4">
                        <i class="fas fa-tags me-2 text-primary"></i>
                        Sales by Category
                        <?php if ($category_filter): ?>
                        <span class="badge bg-primary ms-2"><?php echo ucfirst($category_filter); ?></span>
                        <?php endif; ?>
                    </h5>
                    
                    <?php if (empty($category_sales)): ?>
                    <div class="text-center py-5">
                        <div class="alert alert-info">No sales data found for the selected period and filters.</div>
                    </div>
                    <?php else: ?>
                    <div class="chart-container mb-4">
                        <canvas id="categoryChart"></canvas>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-hover data-table">
                            <thead>
                                <tr>
                                    <th>Category</th>
                                    <th>Quantity Sold</th>
                                    <th>Total Revenue</th>
                                    <th>Avg Price</th>
                                    <th>Percentage</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $total_revenue = array_sum(array_column($category_sales, 'revenue'));
                                foreach ($category_sales as $category):
                                    $percentage = $total_revenue > 0 ? ($category['revenue'] / $total_revenue * 100) : 0;
                                ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars(ucfirst($category['category'])); ?></strong>
                                    </td>
                                    <td><?php echo $category['quantity_sold']; ?></td>
                                    <td class="text-success">
                                        <strong>KES <?php echo number_format($category['revenue'], 2); ?></strong>
                                    </td>
                                    <td>
                                        KES <?php echo number_format($category['avg_price'], 2); ?>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="progress progress-thin flex-grow-1 me-2">
                                                <div class="progress-bar bg-primary" 
                                                     style="width: <?php echo $percentage; ?>%"></div>
                                            </div>
                                            <span><?php echo number_format($percentage, 1); ?>%</span>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="table-light">
                                    <td><strong>Total</strong></td>
                                    <td><strong><?php echo array_sum(array_column($category_sales, 'quantity_sold')); ?></strong></td>
                                    <td class="text-success"><strong>KES <?php echo number_format($total_revenue, 2); ?></strong></td>
                                    <td></td>
                                    <td><strong>100%</strong></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    
                    <script>
                        // Category Sales Chart
                        const categoryCtx = document.getElementById('categoryChart').getContext('2d');
                        const categoryChart = new Chart(categoryCtx, {
                            type: 'bar',
                            data: {
                                labels: [<?php 
                                    foreach ($category_sales as $cat) {
                                        echo "'" . ucfirst($cat['category']) . "',";
                                    }
                                ?>],
                                datasets: [{
                                    label: 'Revenue',
                                    data: [<?php 
                                        foreach ($category_sales as $cat) {
                                            echo $cat['revenue'] . ",";
                                        }
                                    ?>],
                                    backgroundColor: [
                                        '#D35400', '#27AE60', '#3498DB', '#F39C12', 
                                        '#E74C3C', '#9B59B6', '#1ABC9C', '#34495E'
                                    ],
                                    borderColor: '#fff',
                                    borderWidth: 1
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    legend: {
                                        display: false
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        ticks: {
                                            callback: function(value) {
                                                return 'KES ' + value.toLocaleString();
                                            }
                                        }
                                    }
                                }
                            }
                        });
                    </script>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <!-- Customer Reports -->
                <?php if ($report_type === 'customers'): ?>
                <div class="report-card">
                    <h5 class="mb-4">
                        <i class="fas fa-users me-2 text-primary"></i>
                        Customer Reports
                        <span class="badge bg-primary ms-2"><?php echo count($customer_reports); ?> Customers</span>
                    </h5>
                    
                    <?php if (empty($customer_reports)): ?>
                    <div class="alert alert-info">No customer data found.</div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover data-table">
                            <thead>
                                <tr>
                                    <th>Customer</th>
                                    <th>Contact</th>
                                    <th>Total Orders</th>
                                    <th>Total Revenue</th>
                                    <th>Avg Order Value</th>
                                    <th>First Order</th>
                                    <th>Last Order</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($customer_reports as $customer): 
                                    $first_order = !empty($customer['first_order_date']) ? date('M j, Y', strtotime($customer['first_order_date'])) : 'Never';
                                    $last_order = !empty($customer['last_order_date']) ? date('M j, Y', strtotime($customer['last_order_date'])) : 'Never';
                                ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="avatar-placeholder bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-2">
                                                <?php echo strtoupper(substr($customer['name'] ?? 'U', 0, 1)); ?>
                                            </div>
                                            <div>
                                                <strong><?php echo htmlspecialchars($customer['name'] ?? 'Unknown Customer'); ?></strong>
                                                <?php if ($customer['email']): ?>
                                                <br><small class="text-muted"><?php echo htmlspecialchars($customer['email']); ?></small>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ($customer['phone']): ?>
                                        <small><?php echo htmlspecialchars($customer['phone']); ?></small>
                                        <?php else: ?>
                                        <span class="text-muted">N/A</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-primary"><?php echo $customer['total_orders'] ?? 0; ?></span>
                                    </td>
                                    <td class="text-success">
                                        <strong>KES <?php echo number_format($customer['total_revenue'] ?? 0, 2); ?></strong>
                                    </td>
                                    <td>
                                        KES <?php echo number_format($customer['avg_order_value'] ?? 0, 2); ?>
                                    </td>
                                    <td>
                                        <small><?php echo $first_order; ?></small>
                                    </td>
                                    <td>
                                        <small><?php echo $last_order; ?></small>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-primary" 
                                                onclick="viewCustomerDetails(<?php echo $customer['id']; ?>)">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Customer Segmentation Stats -->
                    <div class="row mt-4">
                        <div class="col-md-3">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #e8f5e9; color: var(--secondary);">
                                    <i class="fas fa-user-plus"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php 
                                    $new_customers = array_filter($customer_reports, function($c) {
                                        return !empty($c['first_order_date']) && strtotime($c['first_order_date']) >= strtotime('-30 days');
                                    });
                                    echo count($new_customers);
                                    ?>
                                </h4>
                                <p class="text-muted mb-0">New Customers (30 days)</p>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #e3f2fd; color: var(--primary);">
                                    <i class="fas fa-star"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php 
                                    $vip_customers = array_filter($customer_reports, function($c) {
                                        return ($c['total_revenue'] ?? 0) > 10000; // VIP threshold
                                    });
                                    echo count($vip_customers);
                                    ?>
                                </h4>
                                <p class="text-muted mb-0">VIP Customers</p>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #fff3e0; color: var(--warning);">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php 
                                    $inactive_customers = array_filter($customer_reports, function($c) {
                                        return !empty($c['last_order_date']) && strtotime($c['last_order_date']) < strtotime('-90 days');
                                    });
                                    echo count($inactive_customers);
                                    ?>
                                </h4>
                                <p class="text-muted mb-0">Inactive (>90 days)</p>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #f3e5f5; color: #7b1fa2;">
                                    <i class="fas fa-redo"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php 
                                    $repeat_customers = array_filter($customer_reports, function($c) {
                                        return ($c['total_orders'] ?? 0) > 1;
                                    });
                                    echo count($repeat_customers);
                                    ?>
                                </h4>
                                <p class="text-muted mb-0">Repeat Customers</p>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <!-- Hourly/Daily Performance -->
                <?php if ($report_type === 'hourly'): ?>
                <div class="row">
                    <div class="col-md-8">
                        <div class="report-card">
                            <h5 class="mb-4">
                                <i class="fas fa-clock me-2 text-primary"></i>
                                Hourly Performance
                            </h5>
                            <div class="chart-container">
                                <canvas id="hourlyChart"></canvas>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="report-card">
                            <h5 class="mb-4">
                                <i class="fas fa-calendar-day me-2 text-primary"></i>
                                Daily Performance
                            </h5>
                            <div class="chart-container">
                                <canvas id="dailyChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Hourly Data Table -->
                <div class="report-card mt-4">
                    <h5 class="mb-4">
                        <i class="fas fa-table me-2 text-primary"></i>
                        Hourly Performance Details
                    </h5>
                    <div class="table-responsive">
                        <table class="table table-hover data-table">
                            <thead>
                                <tr>
                                    <th>Hour</th>
                                    <th>Day</th>
                                    <th>Orders</th>
                                    <th>Revenue</th>
                                    <th>Avg Order Value</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($hourly_data)): ?>
                                <?php foreach ($hourly_data as $hour): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo sprintf('%02d:00', $hour['hour']); ?></strong>
                                    </td>
                                    <td><?php echo $hour['day_name']; ?></td>
                                    <td><?php echo $hour['order_count']; ?></td>
                                    <td class="text-success">KES <?php echo number_format($hour['revenue'], 2); ?></td>
                                    <td>KES <?php echo number_format($hour['avg_order_value'], 2); ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <?php else: ?>
                                <tr>
                                    <td colspan="5" class="text-center text-muted">No hourly data found</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <script>
                    // Hourly Performance Chart
                    const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
                    
                    // Group data by hour for chart
                    const hourlyLabels = Array.from({length: 24}, (_, i) => i + ':00');
                    const hourlyRevenue = Array(24).fill(0);
                    const hourlyOrders = Array(24).fill(0);
                    
                    <?php if (!empty($hourly_data)): ?>
                    <?php foreach ($hourly_data as $hour): ?>
                    hourlyRevenue[<?php echo $hour['hour']; ?>] = <?php echo $hour['revenue']; ?>;
                    hourlyOrders[<?php echo $hour['hour']; ?>] = <?php echo $hour['order_count']; ?>;
                    <?php endforeach; ?>
                    <?php endif; ?>
                    
                    const hourlyChart = new Chart(hourlyCtx, {
                        type: 'line',
                        data: {
                            labels: hourlyLabels,
                            datasets: [{
                                label: 'Revenue',
                                data: hourlyRevenue,
                                borderColor: '#D35400',
                                backgroundColor: 'rgba(211, 84, 0, 0.1)',
                                tension: 0.4,
                                fill: true,
                                yAxisID: 'y'
                            }, {
                                label: 'Orders',
                                data: hourlyOrders,
                                borderColor: '#27AE60',
                                backgroundColor: 'rgba(39, 174, 96, 0.1)',
                                tension: 0.4,
                                fill: true,
                                yAxisID: 'y1'
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            interaction: {
                                mode: 'index',
                                intersect: false,
                            },
                            scales: {
                                y: {
                                    type: 'linear',
                                    display: true,
                                    position: 'left',
                                    beginAtZero: true,
                                    ticks: {
                                        callback: function(value) {
                                            return 'KES ' + value.toLocaleString();
                                        }
                                    }
                                },
                                y1: {
                                    type: 'linear',
                                    display: true,
                                    position: 'right',
                                    beginAtZero: true,
                                    grid: {
                                        drawOnChartArea: false,
                                    },
                                    ticks: {
                                        callback: function(value) {
                                            return value + ' orders';
                                        }
                                    }
                                }
                            }
                        }
                    });
                    
                    // Daily Performance Chart
                    const dailyCtx = document.getElementById('dailyChart').getContext('2d');
                    const dailyChart = new Chart(dailyCtx, {
                        type: 'bar',
                        data: {
                            labels: [<?php 
                                if (!empty($daily_data)) {
                                    foreach ($daily_data as $day) {
                                        echo "'" . $day['day_name'] . "',";
                                    }
                                }
                            ?>],
                            datasets: [{
                                label: 'Revenue',
                                data: [<?php 
                                    if (!empty($daily_data)) {
                                        foreach ($daily_data as $day) {
                                            echo $day['revenue'] . ",";
                                        }
                                    }
                                ?>],
                                backgroundColor: '#D35400',
                                borderColor: '#A04000',
                                borderWidth: 1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    display: false
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    ticks: {
                                        callback: function(value) {
                                            return 'KES ' + value.toLocaleString();
                                        }
                                    }
                                }
                            }
                        }
                    });
                </script>
                <?php endif; ?>
                
                <!-- Delivery Performance Report -->
                <?php if ($report_type === 'delivery'): ?>
                <div class="report-card">
                    <h5 class="mb-4">
                        <i class="fas fa-truck me-2 text-primary"></i>
                        Delivery Performance
                        <span class="badge bg-primary ms-2"><?php echo count($delivery_data); ?> Locations</span>
                    </h5>
                    
                    <?php if (empty($delivery_data)): ?>
                    <div class="alert alert-info">No delivery data found for the selected period.</div>
                    <?php else: ?>
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #e8f5e9; color: var(--secondary);">
                                    <i class="fas fa-truck"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php echo array_sum(array_column($delivery_data, 'order_count')); ?>
                                </h4>
                                <p class="text-muted mb-0">Total Deliveries</p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #e3f2fd; color: var(--primary);">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <h4 class="mb-1">
                                    <?php 
                                    $delivery_times = array_column($delivery_data, 'avg_delivery_time');
                                    $valid_times = array_filter($delivery_times, function($time) {
                                        return $time !== null && $time > 0;
                                    });
                                    $avg_time = !empty($valid_times) ? array_sum($valid_times) / count($valid_times) : 0;
                                    echo number_format($avg_time, 0);
                                    ?> mins
                                </h4>
                                <p class="text-muted mb-0">Avg Delivery Time</p>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="stats-card">
                                <div class="stat-icon" style="background: #fff3e0; color: var(--warning);">
                                    <i class="fas fa-money-bill-wave"></i>
                                </div>
                                <h4 class="mb-1">
                                    KES <?php echo number_format(array_sum(array_column($delivery_data, 'total_delivery_fees')), 2); ?>
                                </h4>
                                <p class="text-muted mb-0">Total Delivery Fees</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="chart-container mb-4">
                        <canvas id="deliveryChart"></canvas>
                    </div>
                    
                    <div class="table-responsive">
                        <table class="table table-hover data-table">
                            <thead>
                                <tr>
                                    <th>Delivery Location</th>
                                    <th>Orders</th>
                                    <th>Revenue</th>
                                    <th>Avg Order Value</th>
                                    <th>Delivery Fees</th>
                                    <th>Avg Delivery Time</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($delivery_data as $delivery): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars($delivery['delivery_location']); ?></strong>
                                    </td>
                                    <td><?php echo $delivery['order_count']; ?></td>
                                    <td class="text-success">KES <?php echo number_format($delivery['revenue'], 2); ?></td>
                                    <td>KES <?php echo number_format($delivery['avg_order_value'], 2); ?></td>
                                    <td>KES <?php echo number_format($delivery['total_delivery_fees'], 2); ?></td>
                                    <td>
                                        <?php if ($delivery['avg_delivery_time']): ?>
                                        <span class="badge <?php echo $delivery['avg_delivery_time'] > 45 ? 'bg-danger' : 'bg-success'; ?>">
                                            <?php echo number_format($delivery['avg_delivery_time'], 0); ?> mins
                                        </span>
                                        <?php else: ?>
                                        <span class="text-muted">N/A</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <script>
                        // Delivery Performance Chart
                        const deliveryCtx = document.getElementById('deliveryChart').getContext('2d');
                        const deliveryChart = new Chart(deliveryCtx, {
                            type: 'bar',
                            data: {
                                labels: [<?php 
                                    foreach ($delivery_data as $del) {
                                        echo "'" . htmlspecialchars(substr($del['delivery_location'], 0, 15)) . "',";
                                    }
                                ?>],
                                datasets: [{
                                    label: 'Orders',
                                    data: [<?php 
                                        foreach ($delivery_data as $del) {
                                            echo $del['order_count'] . ",";
                                        }
                                    ?>],
                                    backgroundColor: '#D35400',
                                    borderColor: '#A04000',
                                    borderWidth: 1
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    legend: {
                                        display: false
                                    }
                                },
                                scales: {
                                    y: {
                                        beginAtZero: true,
                                        ticks: {
                                            callback: function(value) {
                                                return value.toLocaleString();
                                            }
                                        }
                                    }
                                }
                            }
                        });
                    </script>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
                
                <!-- Financial Report -->
                <?php if ($report_type === 'financial'): ?>
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e8f5e9; color: var(--secondary);">
                                <i class="fas fa-money-bill-wave"></i>
                            </div>
                            <h4 class="mb-1">
                                KES <?php echo number_format(array_sum(array_column($financial_data, 'revenue')), 2); ?>
                            </h4>
                            <p class="text-muted mb-0">Total Revenue</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #e3f2fd; color: var(--primary);">
                                <i class="fas fa-cash-register"></i>
                            </div>
                            <h4 class="mb-1">
                                KES <?php echo number_format(array_sum(array_column($financial_data, 'collected_amount')), 2); ?>
                            </h4>
                            <p class="text-muted mb-0">Collected Amount</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stats-card">
                            <div class="stat-icon" style="background: #fff3e0; color: var(--warning);">
                                <i class="fas fa-truck"></i>
                            </div>
                            <h4 class="mb-1">
                                KES <?php 
                                $total_delivery_fees = array_sum(array_column($daily_financial, 'daily_delivery_fees'));
                                echo number_format($total_delivery_fees, 2); 
                                ?>
                            </h4>
                            <p class="text-muted mb-0">Total Delivery Fees</p>
                        </div>
                    </div>
                </div>
                
                <div class="row mb-4">
                    <div class="col-md-8">
                        <div class="report-card">
                            <h5 class="mb-4">
                                <i class="fas fa-chart-line me-2 text-primary"></i>
                                Daily Revenue Trend
                            </h5>
                            <div class="chart-container">
                                <canvas id="dailyFinancialChart"></canvas>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="report-card">
                            <h5 class="mb-4">
                                <i class="fas fa-credit-card me-2 text-primary"></i>
                                Payment Methods
                            </h5>
                            <div class="chart-container">
                                <canvas id="financialPieChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="report-card">
                    <h5 class="mb-4">
                        <i class="fas fa-table me-2 text-primary"></i>
                        Financial Summary by Payment Method
                    </h5>
                    <div class="table-responsive">
                        <table class="table table-hover data-table">
                            <thead>
                                <tr>
                                    <th>Payment Method</th>
                                    <th>Transactions</th>
                                    <th>Total Revenue</th>
                                    <th>Collected Amount</th>
                                    <th>Collection Rate</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($financial_data)): ?>
                                <?php foreach ($financial_data as $financial): 
                                    $collection_rate = $financial['revenue'] > 0 ? ($financial['collected_amount'] / $financial['revenue'] * 100) : 0;
                                ?>
                                <tr>
                                    <td>
                                        <strong><?php echo ucfirst($financial['payment_method']); ?></strong>
                                    </td>
                                    <td><?php echo $financial['transaction_count']; ?></td>
                                    <td class="text-success">KES <?php echo number_format($financial['revenue'], 2); ?></td>
                                    <td class="text-primary">KES <?php echo number_format($financial['collected_amount'], 2); ?></td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="progress progress-thin flex-grow-1 me-2">
                                                <div class="progress-bar bg-success" 
                                                     style="width: <?php echo $collection_rate; ?>%"></div>
                                            </div>
                                            <span><?php echo number_format($collection_rate, 1); ?>%</span>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="table-light">
                                    <td><strong>Total</strong></td>
                                    <td><strong><?php echo array_sum(array_column($financial_data, 'transaction_count')); ?></strong></td>
                                    <td class="text-success"><strong>KES <?php echo number_format(array_sum(array_column($financial_data, 'revenue')), 2); ?></strong></td>
                                    <td class="text-primary"><strong>KES <?php echo number_format(array_sum(array_column($financial_data, 'collected_amount')), 2); ?></strong></td>
                                    <td><strong>
                                        <?php 
                                        $total_revenue_fin = array_sum(array_column($financial_data, 'revenue'));
                                        $total_collected = array_sum(array_column($financial_data, 'collected_amount'));
                                        $total_rate = $total_revenue_fin > 0 ? ($total_collected / $total_revenue_fin * 100) : 0;
                                        echo number_format($total_rate, 1); ?>%
                                    </strong></td>
                                </tr>
                                <?php else: ?>
                                <tr>
                                    <td colspan="5" class="text-center text-muted">No financial data found</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <script>
                    // Daily Financial Chart
                    const dailyFinancialCtx = document.getElementById('dailyFinancialChart').getContext('2d');
                    const dailyFinancialChart = new Chart(dailyFinancialCtx, {
                        type: 'line',
                        data: {
                            labels: [<?php 
                                if (!empty($daily_financial)) {
                                    foreach ($daily_financial as $fin) {
                                        echo "'" . date('M j', strtotime($fin['date'])) . "',";
                                    }
                                }
                            ?>],
                            datasets: [{
                                label: 'Daily Revenue',
                                data: [<?php 
                                    if (!empty($daily_financial)) {
                                        foreach ($daily_financial as $fin) {
                                            echo $fin['daily_revenue'] . ",";
                                        }
                                    }
                                ?>],
                                borderColor: '#D35400',
                                backgroundColor: 'rgba(211, 84, 0, 0.1)',
                                tension: 0.4,
                                fill: true
                            }, {
                                label: 'Daily Orders',
                                data: [<?php 
                                    if (!empty($daily_financial)) {
                                        foreach ($daily_financial as $fin) {
                                            echo $fin['daily_orders'] . ",";
                                        }
                                    }
                                ?>],
                                borderColor: '#27AE60',
                                backgroundColor: 'rgba(39, 174, 96, 0.1)',
                                tension: 0.4,
                                fill: false,
                                yAxisID: 'y1'
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            interaction: {
                                mode: 'index',
                                intersect: false,
                            },
                            scales: {
                                y: {
                                    type: 'linear',
                                    display: true,
                                    position: 'left',
                                    beginAtZero: true,
                                    ticks: {
                                        callback: function(value) {
                                            return 'KES ' + value.toLocaleString();
                                        }
                                    }
                                },
                                y1: {
                                    type: 'linear',
                                    display: true,
                                    position: 'right',
                                    beginAtZero: true,
                                    grid: {
                                        drawOnChartArea: false,
                                    },
                                    ticks: {
                                        callback: function(value) {
                                            return value + ' orders';
                                        }
                                    }
                                }
                            }
                        }
                    });
                    
                    // Financial Pie Chart
                    const financialPieCtx = document.getElementById('financialPieChart').getContext('2d');
                    const financialPieChart = new Chart(financialPieCtx, {
                        type: 'doughnut',
                        data: {
                            labels: [<?php 
                                if (!empty($financial_data)) {
                                    foreach ($financial_data as $fin) {
                                        echo "'" . ucfirst($fin['payment_method']) . "',";
                                    }
                                }
                            ?>],
                            datasets: [{
                                data: [<?php 
                                    if (!empty($financial_data)) {
                                        foreach ($financial_data as $fin) {
                                            echo $fin['revenue'] . ",";
                                        }
                                    }
                                ?>],
                                backgroundColor: [
                                    '#D35400', '#27AE60', '#3498DB', '#F39C12', 
                                    '#E74C3C', '#9B59B6', '#1ABC9C', '#34495E'
                                ]
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'bottom',
                                }
                            }
                        }
                    });
                </script>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Export/Print Scripts -->
    <script>
        function exportReport() {
            const params = new URLSearchParams(window.location.search);
            params.set('export', 'csv');
            window.location.href = window.location.pathname + '?' + params.toString();
        }
        
        function printReport() {
            window.print();
        }
        
        function viewCustomerDetails(customerId) {
            // Implement customer detail modal
            alert('View customer details for ID: ' + customerId);
            // In production: window.location.href = 'customer_details.php?id=' + customerId;
        }
        
        // Initialize tooltips
        document.addEventListener('DOMContentLoaded', function() {
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });
    </script>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>