<?php
// api/menu.php - Dual Outlet Menu API with Size Variations

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

require_once __DIR__ . '/../config/database.php';

$database = new Database();
$conn = $database->getConnection();

// Get outlet parameter from query string
$outlet = isset($_GET['outlet']) ? strtolower(trim($_GET['outlet'])) : 'sushi';

// Validate outlet
$valid_outlets = ['sushi', 'milano'];
if (!in_array($outlet, $valid_outlets)) {
    $outlet = 'sushi'; // Default to sushi
}

// Different delivery locations per outlet
$delivery_locations_by_outlet = [
    'sushi' => [
        ['name' => 'Bamburi', 'cost' => 0],
        ['name' => 'Utange', 'cost' => 100],
        ['name' => 'Kiembeni', 'cost' => 150],
        ['name' => 'Shanzu', 'cost' => 200],
        ['name' => 'Mtwapa', 'cost' => 300],
        ['name' => 'Serena', 'cost' => 300],
        ['name' => 'Nyali', 'cost' => 300],
        ['name' => 'Kisauni', 'cost' => 150],
        ['name' => 'Tudor', 'cost' => 400],
        ['name' => 'Mombasa Island', 'cost' => 500]
    ],
    'milano' => [
        ['name' => 'Bamburi', 'cost' => 0],
        ['name' => 'Utange', 'cost' => 100],
        ['name' => 'Kiembeni', 'cost' => 150],
        ['name' => 'Shanzu', 'cost' => 200],
        ['name' => 'Mtwapa', 'cost' => 300],
        ['name' => 'Serena', 'cost' => 300],
        ['name' => 'Nyali', 'cost' => 300],
        ['name' => 'Kisauni', 'cost' => 150],
        ['name' => 'Tudor', 'cost' => 400],
        ['name' => 'Mombasa Island', 'cost' => 500]
    ]
];

// Cache settings - separate cache for each outlet
$cache_dir = __DIR__ . '/../cache/';
$cache_file = $cache_dir . 'menu_cache_' . $outlet . '.json';
$cache_time = 300; // 5 minutes

// Ensure cache directory exists
if (!file_exists($cache_dir)) {
    mkdir($cache_dir, 0777, true);
}

// Check cache first
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
    readfile($cache_file);
    exit;
}

try {
    // Check if outlet column exists
    $column_check = $conn->query("SHOW COLUMNS FROM menu_items LIKE 'outlet'");
    $has_outlet_column = $column_check->rowCount() > 0;
    
    // Check if product_variations table exists
    $variations_table_check = $conn->query("SHOW TABLES LIKE 'product_variations'");
    $has_variations_table = $variations_table_check->rowCount() > 0;
    
    // Get active menu items for specific outlet
    if ($has_outlet_column) {
        // Use outlet column if it exists
        $stmt = $conn->prepare("SELECT * FROM menu_items 
                               WHERE is_active = 1 
                               AND outlet = :outlet
                               ORDER BY category, sort_order, name");
        $stmt->execute([':outlet' => $outlet]);
    } else {
        // Fallback: all items (for backward compatibility)
        $stmt = $conn->prepare("SELECT * FROM menu_items 
                               WHERE is_active = 1 
                               ORDER BY category, sort_order, name");
        $stmt->execute();
    }
    
    $menu_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // If we have a variations table AND it's the Milano outlet, fetch variations for pizza items
    if ($has_variations_table && $outlet === 'milano') {
        $final_menu_items = [];
        
        foreach ($menu_items as $item) {
            // Check if this item has variations (only for pizzas)
            if (strtolower($item['category']) === 'pizza') {
                // Get all active variations for this product
                $var_stmt = $conn->prepare("
                    SELECT * FROM product_variations 
                    WHERE product_id = :product_id 
                    AND is_active = 1
                    ORDER BY 
                        CASE size 
                            WHEN 'small' THEN 1
                            WHEN 'medium' THEN 2
                            WHEN 'large' THEN 3
                            WHEN 'giant' THEN 4
                            ELSE 5
                        END
                ");
                $var_stmt->execute([':product_id' => $item['id']]);
                $variations = $var_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (count($variations) > 0) {
                    // Format variations for frontend
                    $size_variations = [];
                    foreach ($variations as $variation) {
                        $size_variations[$variation['size']] = [
                            'variation_id' => $variation['id'],
                            'price' => floatval($variation['price']),
                            'size' => $variation['size']
                        ];
                    }
                    
                    // Add variations to the item
                    $item['has_variations'] = true;
                    $item['variations'] = $size_variations;
                    
                    // Set default price to smallest size
                    $first_variation = reset($size_variations);
                    $item['default_price'] = $first_variation['price'];
                    $item['default_size'] = $first_variation['size'];
                    
                    // Add to final items
                    $final_menu_items[] = $item;
                } else {
                    // Pizza has no variations (fallback to old system)
                    $item['has_variations'] = false;
                    $item['price'] = floatval($item['price']);
                    $final_menu_items[] = $item;
                }
            } else {
                // Non-pizza items keep original structure
                $item['has_variations'] = false;
                $item['price'] = floatval($item['price']);
                $final_menu_items[] = $item;
            }
        }
        
        $menu_items = $final_menu_items;
    } else {
        // No variations table or not Milano outlet, use original structure
        foreach ($menu_items as &$item) {
            $item['price'] = floatval($item['price']);
            $item['has_variations'] = false;
        }
    }
    
    // Get unique active categories for this outlet
    if ($has_outlet_column) {
        $stmt = $conn->prepare("SELECT DISTINCT category FROM menu_items 
                               WHERE is_active = 1 
                               AND outlet = :outlet
                               AND category IS NOT NULL 
                               AND category != ''
                               ORDER BY category");
        $stmt->execute([':outlet' => $outlet]);
    } else {
        $stmt = $conn->prepare("SELECT DISTINCT category FROM menu_items 
                               WHERE is_active = 1 
                               AND category IS NOT NULL 
                               AND category != ''
                               ORDER BY category");
        $stmt->execute();
    }
    
    $categories = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'category');
    
    // Get delivery locations for this outlet
    $delivery_locations = $delivery_locations_by_outlet[$outlet] ?? $delivery_locations_by_outlet['sushi'];
    
    $response = [
        'success' => true,
        'outlet' => $outlet,
        'menu_items' => $menu_items,
        'categories' => $categories,
        'delivery_locations' => $delivery_locations,
        'has_variations_table' => $has_variations_table,
        'generated_at' => time(),
        'cache_file' => basename($cache_file)
    ];
    
    // Save to cache
    file_put_contents($cache_file, json_encode($response, JSON_PRETTY_PRINT));
    
    echo json_encode($response);
    
} catch (Exception $e) {
    // Return fallback menu data if database fails
    $fallback_data = getFallbackMenuData($outlet);
    
    echo json_encode([
        'success' => false,
        'outlet' => $outlet,
        'message' => 'Error fetching menu: ' . $e->getMessage(),
        'menu_items' => $fallback_data['menu_items'],
        'categories' => $fallback_data['categories'],
        'delivery_locations' => $delivery_locations_by_outlet[$outlet] ?? $delivery_locations_by_outlet['sushi'],
        'is_fallback' => true
    ]);
}

// Fallback menu data function - UPDATED WITH VARIATIONS
function getFallbackMenuData($outlet) {
    if ($outlet === 'milano') {
        // Milano fallback menu with size variations
        return [
            'menu_items' => [
                [
                    'id' => 1,
                    'name' => 'MARGHERITA',
                    'category' => 'pizza',
                    'description' => 'Pizza sauce, tomatoes, mozzarella, basil, oregano',
                    'has_variations' => true,
                    'variations' => [
                        'small' => ['variation_id' => 101, 'price' => 450, 'size' => 'small'],
                        'medium' => ['variation_id' => 102, 'price' => 650, 'size' => 'medium'],
                        'large' => ['variation_id' => 103, 'price' => 850, 'size' => 'large'],
                        'giant' => ['variation_id' => 104, 'price' => 1050, 'size' => 'giant']
                    ],
                    'default_price' => 450,
                    'default_size' => 'small',
                    'is_active' => 1
                ],
                [
                    'id' => 2,
                    'name' => 'VEG PIZZA',
                    'category' => 'pizza',
                    'description' => 'Chef\'s 100% fresh tomato sauce, mozzarella, tomatoes, sweet pepper, onion',
                    'has_variations' => true,
                    'variations' => [
                        'small' => ['variation_id' => 105, 'price' => 450, 'size' => 'small'],
                        'medium' => ['variation_id' => 106, 'price' => 650, 'size' => 'medium'],
                        'large' => ['variation_id' => 107, 'price' => 850, 'size' => 'large'],
                        'giant' => ['variation_id' => 108, 'price' => 1050, 'size' => 'giant']
                    ],
                    'default_price' => 450,
                    'default_size' => 'small',
                    'is_active' => 1
                ],
                [
                    'id' => 3,
                    'name' => 'CHICKEN MUSHROOM',
                    'category' => 'pizza',
                    'description' => 'BBQ chicken, mushroom, pizza sauce, onion and peppers',
                    'has_variations' => true,
                    'variations' => [
                        'small' => ['variation_id' => 109, 'price' => 500, 'size' => 'small'],
                        'medium' => ['variation_id' => 110, 'price' => 700, 'size' => 'medium'],
                        'large' => ['variation_id' => 111, 'price' => 900, 'size' => 'large'],
                        'giant' => ['variation_id' => 112, 'price' => 1100, 'size' => 'giant']
                    ],
                    'default_price' => 500,
                    'default_size' => 'small',
                    'is_active' => 1
                ],
                [
                    'id' => 18,
                    'name' => 'SPAGHETTI BOLOGNAISE',
                    'category' => 'pasta',
                    'description' => 'Classic Italian pasta with rich meat sauce',
                    'price' => 700,
                    'has_variations' => false,
                    'is_active' => 1
                ],
                [
                    'id' => 19,
                    'name' => 'PENNE ARRABIATA',
                    'category' => 'pasta',
                    'description' => 'Spicy tomato sauce with penne pasta',
                    'price' => 700,
                    'has_variations' => false,
                    'is_active' => 1
                ]
            ],
            'categories' => ['pizza', 'pasta']
        ];
    } else {
        // Sushi fallback menu (no size variations needed)
        return [
            'menu_items' => [
                [
                    'id' => 1,
                    'name' => 'TEKKA MAKI',
                    'category' => 'sushi',
                    'description' => 'A timeless sushi classic — fresh, ruby-red tuna wrapped in seasoned sushi rice and crisp nori seaweed',
                    'price' => 300,
                    'has_variations' => false,
                    'is_active' => 1
                ],
                [
                    'id' => 2,
                    'name' => 'AVOCADO MAKI',
                    'category' => 'sushi',
                    'description' => 'Smooth, creamy avocado wrapped in perfectly seasoned sushi rice and crisp nori seaweed',
                    'price' => 800,
                    'has_variations' => false,
                    'is_active' => 1
                ]
            ],
            'categories' => ['sushi', 'noodles', 'wings']
        ];
    }
}
?>