<?php
// api/pizzas.php - Pizza-specific API with categories and sizes

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

require_once __DIR__ . '/../config/database.php';

$database = new Database();
$conn = $database->getConnection();

// Only Milano outlet for pizzas
$outlet = 'milano';

// Cache settings
$cache_dir = __DIR__ . '/../cache/';
$cache_file = $cache_dir . 'pizza_cache.json';
$cache_time = 300; // 5 minutes

// Ensure cache directory exists
if (!file_exists($cache_dir)) {
    mkdir($cache_dir, 0777, true);
}

// Check cache first
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
    readfile($cache_file);
    exit;
}

try {
    // Get pizza categories (only active ones)
    $stmt = $conn->prepare("
        SELECT * FROM pizza_categories 
        WHERE is_active = 1 
        ORDER BY display_order
    ");
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get all active pizzas with their categories
    $stmt = $conn->prepare("
        SELECT 
            mi.id,
            mi.name,
            mi.description,
            mi.price as base_price,
            mi.sizes,
            mi.image_url,
            pc.id as category_id,
            pc.name as category_name,
            pc.slug as category_slug,
            pc.display_order as category_order
        FROM menu_items mi
        LEFT JOIN pizza_categories pc ON mi.pizza_category_id = pc.id
        WHERE mi.category = 'pizza' 
        AND mi.outlet = :outlet
        AND mi.is_active = 1
        AND pc.is_active = 1
        ORDER BY pc.display_order, mi.name
    ");
    $stmt->execute([':outlet' => $outlet]);
    $pizzas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Process each pizza
    $processedPizzas = [];
    foreach ($pizzas as $pizza) {
        // Parse sizes JSON
        $sizes = [];
        if (!empty($pizza['sizes'])) {
            $sizesData = json_decode($pizza['sizes'], true);
            if ($sizesData) {
                $sizeOrder = ['small' => 1, 'medium' => 2, 'large' => 3, 'giant' => 4];
                
                foreach ($sizesData as $size => $price) {
                    if ($price > 0) {
                        $sizes[] = [
                            'size' => $size,
                            'size_display' => ucfirst($size),
                            'price' => floatval($price)
                        ];
                    }
                }
                
                // Sort sizes by order
                usort($sizes, function($a, $b) use ($sizeOrder) {
                    return ($sizeOrder[$a['size']] ?? 99) <=> ($sizeOrder[$b['size']] ?? 99);
                });
            }
        }
        
        // If no sizes, use base price as small size
        if (empty($sizes) && $pizza['base_price'] > 0) {
            $sizes[] = [
                'size' => 'small',
                'size_display' => 'Small',
                'price' => floatval($pizza['base_price'])
            ];
        }
        
        // Handle image URL
        $image_url = '';
        if (!empty($pizza['image_url'])) {
            if (strpos($pizza['image_url'], 'http') === 0) {
                $image_url = $pizza['image_url'];
            } else {
                $image_url = 'images/' . $pizza['image_url'];
            }
        }
        
        $processedPizzas[] = [
            'id' => $pizza['id'],
            'name' => $pizza['name'],
            'description' => $pizza['description'],
            'image_url' => $image_url,
            'category_name' => $pizza['category_name'] ?? 'Pizzas',
            'category_slug' => $pizza['category_slug'] ?? 'pizzas',
            'sizes' => $sizes
        ];
    }
    
    $response = [
        'success' => true,
        'outlet' => $outlet,
        'categories' => $categories,
        'pizzas' => $processedPizzas,
        'generated_at' => time(),
        'total_pizzas' => count($processedPizzas),
        'total_categories' => count($categories)
    ];
    
    // Save to cache
    file_put_contents($cache_file, json_encode($response, JSON_PRETTY_PRINT));
    
    echo json_encode($response);
    
} catch (Exception $e) {
    // Return empty response on error
    echo json_encode([
        'success' => false,
        'message' => 'Error fetching pizza data: ' . $e->getMessage(),
        'categories' => [],
        'pizzas' => [],
        'is_fallback' => true
    ]);
}
?>