<?php
// feedback.php - Customer Feedback System for SUSHI-BAR

// Initialize variables
$success_message = "";
$error_message = "";
$name = "";
$email = "";
$phone = "";
$order_number = "";
$rating = "";
$feedback = "";
$feedback_type = "";

// Process form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitize and validate input
    $name = htmlspecialchars(trim($_POST['name']));
    $email = filter_var(trim($_POST['email']), FILTER_SANITIZE_EMAIL);
    $phone = htmlspecialchars(trim($_POST['phone']));
    $order_number = htmlspecialchars(trim($_POST['order_number']));
    $rating = intval($_POST['rating']);
    $feedback = htmlspecialchars(trim($_POST['feedback']));
    $feedback_type = htmlspecialchars(trim($_POST['feedback_type']));
    $contact_consent = isset($_POST['contact_consent']) ? 1 : 0;
    $submitted_at = date('Y-m-d H:i:s');
    
    // Validate required fields
    if (empty($name) || empty($feedback) || empty($feedback_type) || empty($rating)) {
        $error_message = "Please fill in all required fields (Name, Feedback Type, Rating, and Feedback).";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL) && !empty($email)) {
        $error_message = "Please enter a valid email address.";
    } else {
        // Prepare data for admin page
        $feedback_data = [
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'order_number' => $order_number,
            'rating' => $rating,
            'feedback' => $feedback,
            'feedback_type' => $feedback_type,
            'contact_consent' => $contact_consent,
            'submitted_at' => $submitted_at,
            'status' => 'pending'
        ];
        
        // Save to JSON file in admin directory
        $result = saveFeedbackToAdmin($feedback_data);
        
        if ($result) {
            $success_message = "Thank you for your feedback! We appreciate your input and will review it shortly.";
            
            // Reset form
            $name = $email = $phone = $order_number = $feedback = "";
            $rating = 5;
            $feedback_type = "";
        } else {
            $error_message = "Sorry, there was an error submitting your feedback. Please try again.";
        }
    }
}

// Function to save feedback to admin directory
function saveFeedbackToAdmin($feedback_data) {
    $admin_dir = 'admin/';
    
    // Create admin directory if it doesn't exist
    if (!file_exists($admin_dir)) {
        mkdir($admin_dir, 0755, true);
    }
    
    $feedback_file = $admin_dir . 'customer_feedback.php';
    
    // Initialize feedback array
    $all_feedback = [];
    
    // Read existing feedback if file exists
    if (file_exists($feedback_file)) {
        // Read the file, but skip the PHP opening tag
        $content = file_get_contents($feedback_file);
        // Remove the PHP opening tag and get the JSON part
        $json_start = strpos($content, '[');
        if ($json_start !== false) {
            $json_content = substr($content, $json_start);
            $all_feedback = json_decode($json_content, true);
            if (!is_array($all_feedback)) {
                $all_feedback = [];
            }
        }
    }
    
    // Add new feedback with unique ID
    $feedback_data['id'] = uniqid() . '_' . time();
    $all_feedback[] = $feedback_data;
    
    // Save to file
    $file_content = '<?php
/**
 * SUSHI-BAR Customer Feedback
 * This file stores customer feedback submissions
 * Access this page through admin panel with proper authentication
 */
header("Content-Type: application/json");
?>';
    $file_content .= "\n\n" . json_encode($all_feedback, JSON_PRETTY_PRINT);
    
    return file_put_contents($feedback_file, $file_content);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Feedback - SUSHI-BAR</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        :root {
            --primary: #e53935;
            --primary-dark: #c62828;
            --secondary: #25D366;
            --light: #f5f5f5;
            --dark: #333;
            --gray: #777;
            --light-gray: #f9f9f9;
            --border: #ddd;
            --shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        body {
            background-color: #f8f8f8;
            color: var(--dark);
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        /* Header */
        header {
            background: linear-gradient(to right, var(--primary), var(--primary-dark));
            color: white;
            padding: 20px 0;
            box-shadow: var(--shadow);
            margin-bottom: 40px;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .logo img {
            height: 50px;
            width: auto;
        }
        
        .logo h1 {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .logo span {
            color: #ffeb3b;
        }
        
        .back-btn {
            background: white;
            color: var(--primary);
            padding: 10px 20px;
            border-radius: 30px;
            text-decoration: none;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s;
        }
        
        .back-btn:hover {
            background: #f0f0f0;
            transform: translateY(-2px);
        }
        
        /* Main Content */
        .main-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 40px;
            margin-bottom: 60px;
        }
        
        @media (max-width: 768px) {
            .main-content {
                grid-template-columns: 1fr;
            }
        }
        
        /* Feedback Form */
        .feedback-form-container {
            background: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: var(--shadow);
        }
        
        .section-title {
            font-size: 1.8rem;
            margin-bottom: 25px;
            padding-bottom: 10px;
            border-bottom: 3px solid var(--primary);
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .form-label.required::after {
            content: " *";
            color: var(--primary);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid var(--border);
            border-radius: 5px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 2px rgba(229, 57, 53, 0.2);
        }
        
        textarea.form-control {
            min-height: 120px;
            resize: vertical;
        }
        
        /* Rating Stars */
        .rating-container {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .star-rating {
            display: flex;
            gap: 5px;
            font-size: 2rem;
        }
        
        .star-rating input {
            display: none;
        }
        
        .star-rating label {
            cursor: pointer;
            color: #ddd;
            transition: color 0.3s;
        }
        
        .star-rating input:checked ~ label,
        .star-rating label:hover,
        .star-rating label:hover ~ label {
            color: #ff9800;
        }
        
        .star-rating input:checked + label {
            color: #ff9800;
        }
        
        .rating-labels {
            display: flex;
            justify-content: space-between;
            font-size: 0.9rem;
            color: var(--gray);
        }
        
        /* Feedback Type */
        .feedback-type-container {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 10px;
        }
        
        .feedback-type {
            position: relative;
        }
        
        .feedback-type input {
            display: none;
        }
        
        .feedback-type label {
            display: block;
            padding: 15px;
            border: 2px solid var(--border);
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
            text-align: center;
            font-weight: 600;
        }
        
        .feedback-type input:checked + label {
            border-color: var(--primary);
            background-color: rgba(229, 57, 53, 0.1);
            color: var(--primary);
        }
        
        .feedback-type label:hover {
            border-color: var(--primary);
        }
        
        .feedback-type i {
            display: block;
            font-size: 2rem;
            margin-bottom: 8px;
        }
        
        /* Checkbox */
        .checkbox-container {
            display: flex;
            align-items: flex-start;
            gap: 10px;
        }
        
        .checkbox-container input {
            margin-top: 5px;
        }
        
        .checkbox-text {
            font-size: 0.9rem;
            color: var(--gray);
        }
        
        /* Submit Button */
        .submit-btn {
            background: var(--primary);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.3s;
            display: flex;
            align-items: center;
            gap: 10px;
            width: 100%;
            justify-content: center;
        }
        
        .submit-btn:hover {
            background: var(--primary-dark);
        }
        
        /* Info Section */
        .info-container {
            background: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: var(--shadow);
        }
        
        .info-box {
            background: var(--light-gray);
            border-left: 4px solid var(--primary);
            padding: 20px;
            margin-bottom: 30px;
            border-radius: 0 5px 5px 0;
        }
        
        .info-box h3 {
            margin-bottom: 10px;
            color: var(--primary);
        }
        
        .contact-info {
            margin-top: 30px;
        }
        
        .contact-item {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
            padding: 10px 0;
            border-bottom: 1px solid var(--border);
        }
        
        .contact-item i {
            color: var(--primary);
            font-size: 1.2rem;
            width: 30px;
        }
        
        /* Messages */
        .message {
            padding: 15px 20px;
            border-radius: 5px;
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 15px;
            animation: fadeIn 0.5s ease-out;
        }
        
        .success-message {
            background-color: rgba(46, 125, 50, 0.1);
            border: 1px solid #2e7d32;
            color: #2e7d32;
        }
        
        .error-message {
            background-color: rgba(229, 57, 53, 0.1);
            border: 1px solid var(--primary);
            color: var(--primary);
        }
        
        /* Footer */
        footer {
            background: var(--dark);
            color: white;
            padding: 30px 0;
            margin-top: 60px;
        }
        
        .footer-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 20px;
        }
        
        .copyright {
            text-align: center;
            width: 100%;
            margin-top: 20px;
        }
        
        /* Animations */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header>
        <div class="container header-content">
            <div class="logo">
                <img src="https://aromacafe.co.ke/blaqpremium/images/sushilogo.png" 
                     alt="SUSHI-BAR Logo">
                <h1>SUSHI<span>-BAR</span> <span style="font-size: 1rem;">| Feedback</span></h1>
            </div>
            <a href="index.html" class="back-btn">
                <i class="fas fa-arrow-left"></i> Back to Ordering
            </a>
        </div>
    </header>

    <div class="container">
        <!-- Success/Error Messages -->
        <?php if (!empty($success_message)): ?>
            <div class="message success-message">
                <i class="fas fa-check-circle"></i>
                <span><?php echo $success_message; ?></span>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error_message)): ?>
            <div class="message error-message">
                <i class="fas fa-exclamation-circle"></i>
                <span><?php echo $error_message; ?></span>
            </div>
        <?php endif; ?>

        <div class="main-content">
            <!-- Feedback Form -->
            <div class="feedback-form-container">
                <h2 class="section-title">
                    <i class="fas fa-comment-alt"></i> Share Your Feedback
                </h2>
                
                <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="POST">
                    <div class="form-group">
                        <label class="form-label required" for="name">Full Name</label>
                        <input type="text" id="name" name="name" class="form-control" 
                               value="<?php echo htmlspecialchars($name); ?>" 
                               placeholder="Enter your full name" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="email">Email Address</label>
                        <input type="email" id="email" name="email" class="form-control" 
                               value="<?php echo htmlspecialchars($email); ?>" 
                               placeholder="Enter your email (optional)">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="phone">Phone Number</label>
                        <input type="tel" id="phone" name="phone" class="form-control" 
                               value="<?php echo htmlspecialchars($phone); ?>" 
                               placeholder="Enter your phone number (optional)">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="order_number">Order Number</label>
                        <input type="text" id="order_number" name="order_number" class="form-control" 
                               value="<?php echo htmlspecialchars($order_number); ?>" 
                               placeholder="Enter your order number (optional)">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label required">Feedback Type</label>
                        <div class="feedback-type-container">
                            <div class="feedback-type">
                                <input type="radio" id="compliment" name="feedback_type" value="Compliment" 
                                       <?php echo ($feedback_type == 'Compliment') ? 'checked' : ''; ?> required>
                                <label for="compliment">
                                    <i class="fas fa-heart"></i>
                                    <span>Compliment</span>
                                </label>
                            </div>
                            <div class="feedback-type">
                                <input type="radio" id="suggestion" name="feedback_type" value="Suggestion"
                                       <?php echo ($feedback_type == 'Suggestion') ? 'checked' : ''; ?>>
                                <label for="suggestion">
                                    <i class="fas fa-lightbulb"></i>
                                    <span>Suggestion</span>
                                </label>
                            </div>
                            <div class="feedback-type">
                                <input type="radio" id="complaint" name="feedback_type" value="Complaint"
                                       <?php echo ($feedback_type == 'Complaint') ? 'checked' : ''; ?>>
                                <label for="complaint">
                                    <i class="fas fa-exclamation-triangle"></i>
                                    <span>Complaint</span>
                                </label>
                            </div>
                            <div class="feedback-type">
                                <input type="radio" id="other" name="feedback_type" value="Other"
                                       <?php echo ($feedback_type == 'Other') ? 'checked' : ''; ?>>
                                <label for="other">
                                    <i class="fas fa-comment"></i>
                                    <span>Other</span>
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label required">Overall Rating</label>
                        <div class="rating-container">
                            <div class="star-rating">
                                <input type="radio" id="star5" name="rating" value="5" 
                                       <?php echo ($rating == 5 || empty($rating)) ? 'checked' : ''; ?>>
                                <label for="star5" title="Excellent">★</label>
                                
                                <input type="radio" id="star4" name="rating" value="4"
                                       <?php echo ($rating == 4) ? 'checked' : ''; ?>>
                                <label for="star4" title="Good">★</label>
                                
                                <input type="radio" id="star3" name="rating" value="3"
                                       <?php echo ($rating == 3) ? 'checked' : ''; ?>>
                                <label for="star3" title="Average">★</label>
                                
                                <input type="radio" id="star2" name="rating" value="2"
                                       <?php echo ($rating == 2) ? 'checked' : ''; ?>>
                                <label for="star2" title="Poor">★</label>
                                
                                <input type="radio" id="star1" name="rating" value="1"
                                       <?php echo ($rating == 1) ? 'checked' : ''; ?>>
                                <label for="star1" title="Very Poor">★</label>
                            </div>
                            <div class="rating-labels">
                                <span>Very Poor</span>
                                <span>Excellent</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label required" for="feedback">Your Feedback</label>
                        <textarea id="feedback" name="feedback" class="form-control" 
                                  placeholder="Please share your detailed feedback about your experience with SUSHI-BAR..." 
                                  required><?php echo htmlspecialchars($feedback); ?></textarea>
                    </div>
                    
                    <div class="form-group">
                        <div class="checkbox-container">
                            <input type="checkbox" id="contact_consent" name="contact_consent" value="1">
                            <label for="contact_consent" class="checkbox-text">
                                I give permission for SUSHI-BAR to contact me regarding this feedback if needed.
                            </label>
                        </div>
                    </div>
                    
                    <button type="submit" class="submit-btn">
                        <i class="fas fa-paper-plane"></i> Submit Feedback
                    </button>
                </form>
            </div>
            
            <!-- Information Section -->
            <div class="info-container">
                <h2 class="section-title">
                    <i class="fas fa-info-circle"></i> About Feedback
                </h2>
                
                <div class="info-box">
                    <h3>Why Your Feedback Matters</h3>
                    <p>Your feedback helps us improve our food quality, service, and overall customer experience. We value every comment and use it to make SUSHI-BAR better for everyone.</p>
                </div>
                
                <div class="info-box">
                    <h3>What Happens Next?</h3>
                    <p>Our team reviews all feedback daily. For urgent complaints, we aim to respond within 24 hours. Your feedback helps us identify areas for improvement and celebrate what we're doing right.</p>
                </div>
                
                <div class="contact-info">
                    <h3 class="section-title">
                        <i class="fas fa-headset"></i> Need Immediate Help?
                    </h3>
                    
                    <div class="contact-item">
                        <i class="fab fa-whatsapp"></i>
                        <div>
                            <strong>WhatsApp Support</strong><br>
                            <a href="https://wa.me/254790644644" style="color: var(--dark); text-decoration: none;">+254 790 644 644</a>
                        </div>
                    </div>
                    
                    <div class="contact-item">
                        <i class="fas fa-phone"></i>
                        <div>
                            <strong>Call Us</strong><br>
                            +254 790 644 644
                        </div>
                    </div>
                    
                    <div class="contact-item">
                        <i class="fas fa-envelope"></i>
                        <div>
                            <strong>Email</strong><br>
                            support@sushi-bar.com
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Footer -->
    <footer>
        <div class="container">
            <div class="footer-content">
                <div class="footer-logo">
                    <img src="https://aromacafe.co.ke/blaqpremium/images/sushilogo.png" 
                         alt="SUSHI-BAR Logo" 
                         style="height: 50px; width: auto;">
                </div>
                <div class="contact-info" style="color: white;">
                    <div style="margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> <span>Mombasa, Kenya</span></div>
                    <div style="margin-bottom: 10px;"><i class="fas fa-phone"></i> <span>+254 790 644 644</span></div>
                    <div><i class="fab fa-whatsapp"></i> <span>+254 790 644 644</span></div>
                </div>
            </div>
            <div class="copyright">
                <p>© 2025 SUSHI-BAR. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <script>
        // Add interactivity to star ratings
        document.addEventListener('DOMContentLoaded', function() {
            const stars = document.querySelectorAll('.star-rating label');
            
            stars.forEach(star => {
                star.addEventListener('click', function() {
                    const rating = this.previousElementSibling.value;
                    // You can add additional logic here if needed
                });
            });
            
            // Auto-hide success message after 10 seconds
            const successMessage = document.querySelector('.success-message');
            if (successMessage) {
                setTimeout(() => {
                    successMessage.style.transition = 'opacity 0.5s';
                    successMessage.style.opacity = '0';
                    setTimeout(() => {
                        successMessage.style.display = 'none';
                    }, 500);
                }, 10000);
            }
        });
    </script>
</body>
</html>