<?php
// includes/MpesaService.php - FIXED VERSION

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/mpesa_config.php';

class MpesaService {

    private $conn;

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    private function logError($message) {
        $logDir = __DIR__ . '/../logs';
        if (!is_dir($logDir)) mkdir($logDir, 0755, true);

        $file = $logDir . '/mpesa_service.log';
        file_put_contents($file, "[".date('Y-m-d H:i:s')."] $message\n", FILE_APPEND);
    }

    // ----------------------------------------------------------
    // 1. GET ACCESS TOKEN (Production URL)
    // ----------------------------------------------------------
    public function getAccessToken() {

        $credentials = base64_encode(
            MpesaConfig::CONSUMER_KEY . ":" . MpesaConfig::CONSUMER_SECRET
        );

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, MpesaConfig::OAUTH_URL);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Basic $credentials"
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            $err = curl_error($ch);
            $this->logError("Token cURL Error: " . $err);
            curl_close($ch);
            return null;
        }

        curl_close($ch);

        $data = json_decode($response, true);

        if (!empty($data['access_token'])) {
            return $data['access_token'];
        }

        $this->logError("Failed to retrieve access token: $response");
        return null;
    }

    // ----------------------------------------------------------
    // 2. INITIATE STK PUSH - FIXED SUCCESS HANDLING
    // ----------------------------------------------------------
    public function initiateSTKPush($phone, $amount, $orderId, $description = "SUSHI-BAR") {

        $this->logError("Initiating STK Push: phone=$phone amount=$amount order=$orderId");

        $token = $this->getAccessToken();
        if (!$token) {
            return ['success' => false, 'message' => 'Failed to authenticate with M-Pesa'];
        }

        $passwordData = MpesaConfig::generatePassword();

        $payload = [
            "BusinessShortCode" => MpesaConfig::BUSINESS_SHORT_CODE,
            "Password" => $passwordData['password'],
            "Timestamp" => $passwordData['timestamp'],
            "TransactionType" => "CustomerPayBillOnline",
            "Amount" => round($amount),
            "PartyA" => $phone,
            "PartyB" => MpesaConfig::BUSINESS_SHORT_CODE,
            "PhoneNumber" => $phone,
            "CallBackURL" => MpesaConfig::CALLBACK_URL,
            "AccountReference" => "ORDER-" . $orderId,
            "TransactionDesc" => $description
        ];

        $this->logError("STK Payload: " . json_encode($payload));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, MpesaConfig::STK_PUSH_URL);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer $token",
            "Content-Type: application/json"
        ]);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            $this->logError("STK Error: $error");
            return ['success' => false, 'message' => "STK Network error: $error"];
        }

        curl_close($ch);

        $result = json_decode($response, true);

        $this->logError("STK Full Response: " . $response);

        // CHECK FOR SUCCESS - FIXED!
        // M-Pesa returns success in different formats
        $isSuccess = false;
        $checkoutId = null;
        $merchantId = null;
        $successMessage = '';
        
        // Check various success indicators
        if (!empty($result['ResponseCode']) && $result['ResponseCode'] == 0) {
            $isSuccess = true;
            $checkoutId = $result['CheckoutRequestID'] ?? null;
            $merchantId = $result['MerchantRequestID'] ?? null;
            $successMessage = $result['ResponseDescription'] ?? 'STK Push sent successfully';
        } 
        // Check for "Success" in the description
        elseif (!empty($result['ResponseDescription']) && 
                (stripos($result['ResponseDescription'], 'success') !== false ||
                 stripos($result['ResponseDescription'], 'request accepted') !== false)) {
            
            $isSuccess = true;
            $checkoutId = $result['CheckoutRequestID'] ?? null;
            $merchantId = $result['MerchantRequestID'] ?? null;
            $successMessage = $result['ResponseDescription'];
        }
        // Check for CustomerMessage as success indicator
        elseif (!empty($result['CustomerMessage']) && 
                stripos($result['CustomerMessage'], 'success') !== false) {
            
            $isSuccess = true;
            $checkoutId = $result['CheckoutRequestID'] ?? null;
            $merchantId = $result['MerchantRequestID'] ?? null;
            $successMessage = $result['CustomerMessage'];
        }

        if ($isSuccess) {
            $this->logError("STK SUCCESS: CheckoutID=$checkoutId, MerchantID=$merchantId, Message=$successMessage");

            // Save transaction (non-blocking)
            if ($checkoutId) {
                try {
                    $this->saveTransactionLog(
                        $merchantId,
                        $checkoutId,
                        $amount,
                        $phone,
                        $orderId
                    );
                } catch (Exception $e) {
                    $this->logError("SaveTransaction Exception: " . $e->getMessage());
                }

                // Update order checkout id
                try {
                    $this->updateOrderCheckoutId($orderId, $checkoutId);
                } catch (Exception $e) {
                    $this->logError("UpdateOrder Exception: " . $e->getMessage());
                }
            }

            return [
                "success" => true,
                "checkout_id" => $checkoutId,
                "merchant_request_id" => $merchantId,
                "message" => $successMessage,
                "raw_response" => $response
            ];
        }

        // FAILURE
        $errorMsg = $result['errorMessage'] ??
                   $result['ResponseDescription'] ??
                   $result['CustomerMessage'] ??
                   (isset($result['errorCode']) ? 'Error Code: ' . $result['errorCode'] : 'Unknown STK error');

        $this->logError("STK FAILURE: $errorMsg");

        return [
            "success" => false, 
            "message" => $errorMsg, 
            "raw_response" => $response
        ];
    }

    // ----------------------------------------------------------
    // 3. SAVE TRANSACTION LOG
    // ----------------------------------------------------------
    private function saveTransactionLog($merchantRequestId, $checkoutRequestId, $amount, $phone, $orderId) {
        try {
            $sql = "INSERT INTO mpesa_transactions
                    (merchant_request_id, checkout_request_id, amount, phone_number, status, order_id, created_at)
                    VALUES (:m, :c, :a, :p, 'pending', :o, NOW())";

            $stmt = $this->conn->prepare($sql);
            $stmt->execute([
                ':m' => $merchantRequestId,
                ':c' => $checkoutRequestId,
                ':a' => $amount,
                ':p' => $phone,
                ':o' => $orderId
            ]);

        } catch (Exception $e) {
            $this->logError("TransactionLog Error: " . $e->getMessage());
        }
    }

    // ----------------------------------------------------------
    // 4. UPDATE ORDER CHECKOUT ID
    // ----------------------------------------------------------
    private function updateOrderCheckoutId($orderId, $checkoutId) {
        try {
            $sql = "UPDATE orders SET mpesa_checkout_id = :c WHERE id = :id";
            $stmt = $this->conn->prepare($sql);
            $stmt->execute([
                ":c" => $checkoutId,
                ":id" => $orderId
            ]);

        } catch (Exception $e) {
            $this->logError("UpdateOrder Error: " . $e->getMessage());
        }
    }
    
    // ----------------------------------------------------------
    // 5. QUERY STK STATUS (Optional - for debugging)
    // ----------------------------------------------------------
    public function queryStkStatus($checkoutRequestId) {
        try {
            $token = $this->getAccessToken();
            if (!$token) {
                return ['success' => false, 'message' => 'Failed to authenticate'];
            }
            
            $passwordData = MpesaConfig::generatePassword();
            
            $payload = [
                "BusinessShortCode" => MpesaConfig::BUSINESS_SHORT_CODE,
                "Password" => $passwordData['password'],
                "Timestamp" => $passwordData['timestamp'],
                "CheckoutRequestID" => $checkoutRequestId
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, MpesaConfig::QUERY_URL);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer $token",
                "Content-Type: application/json"
            ]);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            curl_close($ch);
            
            $this->logError("Query Response: " . $response);
            
            return json_decode($response, true);
            
        } catch (Exception $e) {
            $this->logError("Query Error: " . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
?>