<?php
// includes/whatsapp_api.php - WhatsApp Integration

class WhatsAppAPI {
    private $api_url;
    private $api_key;
    private $api_secret;
    
    public function __construct() {
        // Configure your WhatsApp API provider (Twilio, WhatsApp Business API, etc.)
        $this->api_url = 'https://api.whatsapp.com/v1/messages';
        $this->api_key = 'YOUR_API_KEY';
        $this->api_secret = 'YOUR_API_SECRET';
        
        // Alternative: Use a service like Twilio
        // $this->account_sid = 'your_account_sid';
        // $this->auth_token = 'your_auth_token';
        // $this->whatsapp_from = 'whatsapp:+14155238886'; // Twilio sandbox number
    }
    
    /**
     * Send WhatsApp message
     * @param string $to Phone number (with country code)
     * @param string $message Message text
     * @return array Result with success status
     */
    public function sendMessage($to, $message) {
        try {
            // Clean phone number (remove non-numeric characters except +)
            $to = preg_replace('/[^0-9+]/', '', $to);
            
            // Add country code if missing
            if (!str_starts_with($to, '+')) {
                $to = '+254' . ltrim($to, '0'); // Kenya default
            }
            
            // Choose your WhatsApp API provider implementation:
            
            // Method 1: Using cURL (generic API)
            return $this->sendViaCurl($to, $message);
            
            // Method 2: Using Twilio (recommended)
            // return $this->sendViaTwilio($to, $message);
            
            // Method 3: Using Vonage
            // return $this->sendViaVonage($to, $message);
            
        } catch (Exception $e) {
            error_log("WhatsApp API Error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to send WhatsApp message: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Send via generic REST API
     */
    private function sendViaCurl($to, $message) {
        $data = [
            'to' => $to,
            'message' => $message,
            'type' => 'text'
        ];
        
        $ch = curl_init($this->api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->api_key
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200) {
            return ['success' => true, 'message' => 'Message sent successfully'];
        } else {
            return [
                'success' => false,
                'message' => 'API returned HTTP ' . $http_code . ': ' . $response
            ];
        }
    }
    
    /**
     * Send via Twilio API (Recommended)
     */
    private function sendViaTwilio($to, $message) {
        require_once __DIR__ . '/../vendor/autoload.php'; // If using Composer
        
        $account_sid = 'your_account_sid';
        $auth_token = 'your_auth_token';
        $from_number = 'whatsapp:+14155238886'; // Twilio sandbox number
        
        $client = new Twilio\Rest\Client($account_sid, $auth_token);
        
        try {
            $message = $client->messages->create(
                'whatsapp:' . $to,
                [
                    'from' => $from_number,
                    'body' => $message
                ]
            );
            
            // Log the message SID for tracking
            error_log("WhatsApp message sent. SID: " . $message->sid);
            
            return [
                'success' => true,
                'message' => 'Message sent successfully',
                'sid' => $message->sid
            ];
            
        } catch (Exception $e) {
            error_log("Twilio Error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Twilio error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Send via Vonage API (Nexmo)
     */
    private function sendViaVonage($to, $message) {
        $basic  = new \Vonage\Client\Credentials\Basic('your_api_key', 'your_api_secret');
        $client = new \Vonage\Client($basic);
        
        try {
            $response = $client->message()->send([
                'to' => $to,
                'from' => 'your_vonage_number',
                'text' => $message,
                'type' => 'text'
            ]);
            
            return [
                'success' => true,
                'message' => 'Message sent successfully',
                'message_id' => $response['messages'][0]['message-id']
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Vonage error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Send via WhatsApp Business Cloud API
     */
    private function sendViaWhatsAppCloudAPI($to, $message) {
        $access_token = 'YOUR_WHATSAPP_CLOUD_API_ACCESS_TOKEN';
        $phone_number_id = 'YOUR_PHONE_NUMBER_ID';
        
        $url = "https://graph.facebook.com/v17.0/{$phone_number_id}/messages";
        
        $data = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'text',
            'text' => ['body' => $message]
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $access_token
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200) {
            $response_data = json_decode($response, true);
            return [
                'success' => true,
                'message' => 'Message sent via WhatsApp Cloud API',
                'message_id' => $response_data['messages'][0]['id']
            ];
        } else {
            return [
                'success' => false,
                'message' => 'WhatsApp Cloud API error: HTTP ' . $http_code
            ];
        }
    }
    
    /**
     * Send template message (for predefined messages)
     */
    public function sendTemplateMessage($to, $template_name, $parameters = []) {
        $data = [
            'to' => $to,
            'template' => [
                'name' => $template_name,
                'language' => ['code' => 'en_US'],
                'components' => []
            ]
        ];
        
        // Add parameters if any
        if (!empty($parameters)) {
            $data['template']['components'][] = [
                'type' => 'body',
                'parameters' => array_map(function($param) {
                    return ['type' => 'text', 'text' => $param];
                }, $parameters)
            ];
        }
        
        // Send via API (implement similar to above methods)
        return $this->sendViaCurl($to, $data);
    }
}
?>